module Scenarios
  ( Scenario(..)
  , uniformForestScenario
  , withWindScenario
  , riverBarrierScenario
  , mixedTerrainScenario
  , uphillScenario
  , allScenarios
  ) where

import Types
import Grid

-- | Named scenario for testing
data Scenario = Scenario
  { scenarioName   :: String
  , scenarioGrid   :: Grid
  , scenarioConfig :: SimConfig
  }

-- | Basic uniform forest scenario
uniformForestScenario :: Int -> Int -> Scenario
uniformForestScenario rows cols = Scenario
  { scenarioName = "Uniform Forest"
  , scenarioGrid = uniformForest 0.8 rows cols
  , scenarioConfig = SimConfig
      { gridRows = rows
      , gridCols = cols
      , wind = Wind 0.0 0.0
      , maxTimeSteps = 500
      , ignitionPoints = [(rows `div` 2, cols `div` 2)]
      , baseProbability = 0.58
      , ignitionThreshold = 0.35
      }
  }

-- | Scenario with strong wind
withWindScenario :: Int -> Int -> Scenario
withWindScenario rows cols = Scenario
  { scenarioName = "Strong Wind"
  , scenarioGrid = uniformForest 0.8 rows cols
  , scenarioConfig = SimConfig
      { gridRows = rows
      , gridCols = cols
      , wind = Wind 0.7 90.0
      , maxTimeSteps = 500
      , ignitionPoints = [(rows `div` 2, cols `div` 4)]
      , baseProbability = 0.58
      , ignitionThreshold = 0.35
      }
  }

-- | Scenario with river barrier
riverBarrierScenario :: Int -> Int -> Scenario
riverBarrierScenario rows cols = Scenario
  { scenarioName = "River Barrier"
  , scenarioGrid = withRiverBarrier rows cols (cols `div` 2)
  , scenarioConfig = SimConfig
      { gridRows = rows
      , gridCols = cols
      , wind = Wind 0.0 0.0
      , maxTimeSteps = 500
      , ignitionPoints = [(rows `div` 2, cols `div` 4)]
      , baseProbability = 0.58
      , ignitionThreshold = 0.35
      }
  }

-- | Scenario with mixed terrain
mixedTerrainScenario :: Int -> Int -> Scenario
mixedTerrainScenario rows cols = Scenario
  { scenarioName = "Mixed Terrain"
  , scenarioGrid = mixedTerrain rows cols
  , scenarioConfig = SimConfig
      { gridRows = rows
      , gridCols = cols
      , wind = Wind 0.3 45.0
      , maxTimeSteps = 500
      , ignitionPoints = [(rows `div` 2, cols `div` 2)]
      , baseProbability = 0.58
      , ignitionThreshold = 0.35
      }
  }

-- | Scenario with elevation gradient (uphill spread)
uphillScenario :: Int -> Int -> Scenario
uphillScenario rows cols = 
  let grid = createGrid rows cols $ \r c ->
        Cell
          { cellState = Unburned
          , cellPosition = (r, c)
          , fuelLevel = 0.8
          , terrainType = Forest
          , elevation = fromIntegral r * 0.2
          , burnSteps = 0
          }
  in Scenario
       { scenarioName = "Uphill Spread"
       , scenarioGrid = grid
       , scenarioConfig = SimConfig
           { gridRows = rows
           , gridCols = cols
           , wind = Wind 0.0 0.0
           , maxTimeSteps = 500
           , ignitionPoints = [(2, cols `div` 2)]
           , baseProbability = 0.58
           , ignitionThreshold = 0.35
           }
       }

-- | All predefined scenarios
allScenarios :: Int -> Int -> [Scenario]
allScenarios rows cols =
  [ uniformForestScenario rows cols
  , withWindScenario rows cols
  , riverBarrierScenario rows cols
  , mixedTerrainScenario rows cols
  , uphillScenario rows cols
  ]