module Grid
  ( createGrid
  , displayGrid
  , displayGridDetailed
  , getCell
  , setCell
  , inBounds
  , getNeighbors8
  , uniformForest
  , mixedTerrain
  , withRiverBarrier
  , countCellsByState
  ) where

import Types
import qualified Data.Vector as V

-- | Create a grid with given dimensions and cell generator function
createGrid :: Int -> Int -> (Int -> Int -> Cell) -> Grid
createGrid rows cols cellGen = 
  V.generate rows $ \r ->
    V.generate cols $ \c ->
      cellGen r c

-- | Get cell at position (with bounds checking)
getCell :: Grid -> (Int, Int) -> Maybe Cell
getCell grid (r, c)
  | inBounds grid (r, c) = Just $ (grid V.! r) V.! c
  | otherwise = Nothing

-- | Set cell at position (with bounds checking)
setCell :: Grid -> (Int, Int) -> Cell -> Grid
setCell grid (r, c) cell
  | inBounds grid (r, c) = 
      let row = grid V.! r
          newRow = row V.// [(c, cell)]
      in grid V.// [(r, newRow)]
  | otherwise = grid

-- | Check if position is within grid bounds
inBounds :: Grid -> (Int, Int) -> Bool
inBounds grid (r, c) = 
  r >= 0 && r < V.length grid && 
  c >= 0 && c < V.length (grid V.! 0)

-- | Get all 8 neighbors of a cell
getNeighbors8 :: Grid -> (Int, Int) -> [(Int, Int)]
getNeighbors8 grid (r, c) = 
  [ (r + dr, c + dc) 
  | dr <- [-1, 0, 1]
  , dc <- [-1, 0, 1]
  , (dr, dc) /= (0, 0)
  , inBounds grid (r + dr, c + dc)
  ]

-- | Count cells by their state
countCellsByState :: Grid -> CellState -> Int
countCellsByState grid state =
  V.sum $ V.map (V.length . V.filter (\c -> cellState c == state)) grid

-- | Display grid as ASCII art (simple view)
displayGrid :: Grid -> String
displayGrid grid = unlines $ V.toList $ V.map displayRow grid
  where
    displayRow row = V.toList $ V.map cellChar row
    cellChar cell = case cellState cell of
      Unburned -> case terrainType cell of
        Forest    -> '.'
        Grassland -> ','
        Water     -> '~'
        Urban     -> '#'
      Burning  -> '*'
      Burned   -> 'x'

-- | Display grid with more detail
displayGridDetailed :: Grid -> String
displayGridDetailed grid = 
  header ++ unlines (zipWith addRowNum [0..] rows)
  where
    rows = V.toList $ V.map displayRow grid
    header = "    " ++ concatMap (\c -> show (c `mod` 10) ++ " ") [0..V.length (grid V.! 0) - 1] ++ "\n"
    addRowNum n row = show n ++ " | " ++ row
    displayRow row = unwords $ V.toList $ V.map cellChar row
    cellChar cell = case cellState cell of
      Unburned -> case terrainType cell of
        Forest    -> " ."
        Grassland -> " ,"
        Water     -> " ~"
        Urban     -> " #"
      Burning  -> " *"
      Burned   -> " x"

-- ========================================
-- Terrain Generation Functions
-- ========================================

-- | Create a uniform forest with given fuel level
uniformForest :: Double -> Int -> Int -> Grid
uniformForest fuel rows cols = createGrid rows cols $ \r c ->
  Cell
    { cellState = Unburned
    , cellPosition = (r, c)
    , fuelLevel = fuel
    , terrainType = Forest
    , elevation = 0.0
    , burnSteps = 0
    }

-- | Create mixed terrain (forest, grassland, clearings)
mixedTerrain :: Int -> Int -> Grid
mixedTerrain rows cols = createGrid rows cols $ \r c ->
  let terrain = if (r + c) `mod` 5 == 0 
                then Grassland
                else if (r * c) `mod` 7 == 0
                then Forest
                else Forest
      fuel = case terrain of
        Forest    -> 0.8
        Grassland -> 0.6
        _         -> 0.0
  in Cell
       { cellState = Unburned
       , cellPosition = (r, c)
       , fuelLevel = fuel
       , terrainType = terrain
       , elevation = fromIntegral (abs (r - rows `div` 2)) * 0.1
       , burnSteps = 0
       }

-- | Create terrain with a vertical river barrier
withRiverBarrier :: Int -> Int -> Int -> Grid
withRiverBarrier rows cols riverCol = createGrid rows cols $ \r c ->
  if c == riverCol || c == riverCol + 1
    then Cell
           { cellState = Unburned
           , cellPosition = (r, c)
           , fuelLevel = 0.0
           , terrainType = Water
           , elevation = 0.0
           , burnSteps = 0
           }
    else Cell
           { cellState = Unburned
           , cellPosition = (r, c)
           , fuelLevel = 0.8
           , terrainType = Forest
           , elevation = 0.0
           , burnSteps = 0
           }