module FireRules
  ( calculateIgnitionProbability
  , shouldIgnite
  , updateCellState
  , getWindFactor
  , getSlopeFactor
  ) where

import Types
import Grid
import System.Random (Random, randomR, RandomGen)

-- | Calculate ignition probability for a cell given its neighbors
calculateIgnitionProbability :: SimConfig -> Cell -> [(Int, Int)] -> Grid -> Double
calculateIgnitionProbability config cell neighborPositions grid =
  let burningNeighbors = filter isBurning $ mapMaybe (getCell grid) neighborPositions
  in if null burningNeighbors
     then 0.0
     else
       let baseProb = baseProbability config
           fuelFactor = fuelLevel cell
           windFactor = avgWindFactor config cell burningNeighbors grid neighborPositions
           slopeFactor = avgSlopeFactor cell burningNeighbors grid neighborPositions
       in baseProb * fuelFactor * windFactor * slopeFactor

-- Helper to check if a cell is burning
isBurning :: Cell -> Bool
isBurning c = cellState c == Burning

-- Helper for safe list operations
mapMaybe :: (a -> Maybe b) -> [a] -> [b]
mapMaybe _ [] = []
mapMaybe f (x:xs) = case f x of
  Nothing -> mapMaybe f xs
  Just y  -> y : mapMaybe f xs

-- | Calculate wind factor contribution
avgWindFactor :: SimConfig -> Cell -> [Cell] -> Grid -> [(Int, Int)] -> Double
avgWindFactor config targetCell burningCells grid neighborPositions =
  if null burningCells
    then 1.0
    else
      let windSpeed' = windSpeed (wind config)
          windDir = windDirection (wind config)
          factors = map (getWindFactor windSpeed' windDir targetCell) burningCells
      in if null factors then 1.0 else sum factors / fromIntegral (length factors)

-- | Get wind factor for a specific burning neighbor
getWindFactor :: Double -> Double -> Cell -> Cell -> Double
getWindFactor windSpeed' windDir targetCell burningCell =
  let (tr, tc) = cellPosition targetCell
      (br, bc) = cellPosition burningCell
      -- Calculate angle from burning cell to target
      dx = fromIntegral (tc - bc)
      dy = fromIntegral (tr - br)
      angleToTarget = atan2 dy dx * 180.0 / pi
      -- Normalize to 0-360
      normalizedAngle = if angleToTarget < 0 then angleToTarget + 360 else angleToTarget
      -- Calculate difference from wind direction
      angleDiff = abs (normalizedAngle - windDir)
      angleDiff' = if angleDiff > 180 then 360 - angleDiff else angleDiff
      -- Wind factor: higher when fire spreads downwind
      windEffect = cos (angleDiff' * pi / 180.0)
  in 1.0 + windSpeed' * max 0 windEffect

-- | Calculate average slope factor
avgSlopeFactor :: Cell -> [Cell] -> Grid -> [(Int, Int)] -> Double
avgSlopeFactor targetCell burningCells grid neighborPositions =
  if null burningCells
    then 1.0
    else
      let factors = map (getSlopeFactor targetCell) burningCells
      in if null factors then 1.0 else sum factors / fromIntegral (length factors)

-- | Get slope factor for a specific burning neighbor
getSlopeFactor :: Cell -> Cell -> Double
getSlopeFactor targetCell burningCell =
  let elevDiff = elevation targetCell - elevation burningCell
      slopeCoeff = 0.3
  in if elevDiff > 0  -- Uphill
     then 1.0 + slopeCoeff * elevDiff
     else 1.0  -- Downhill or flat

-- | Determine if a cell should ignite based on probability and random value
shouldIgnite :: RandomGen g => SimConfig -> Double -> g -> (Bool, g)
shouldIgnite config prob gen =
  let threshold = ignitionThreshold config
      (randomVal, gen') = randomR (0.0 :: Double, 1.0 :: Double) gen
      -- Add random factor between 0.8 and 1.2
      (randomFactor, gen'') = randomR (0.8 :: Double, 1.2 :: Double) gen'
      finalProb = prob * randomFactor
  in (finalProb > threshold, gen'')

-- | Update a single cell's state based on fire rules
updateCellState :: Cell -> Int -> Cell
updateCellState cell maxBurnSteps =
  case cellState cell of
    Unburned -> cell  -- Will be updated by ignition logic
    Burning ->
      let newBurnSteps = burnSteps cell + 1
      in if newBurnSteps >= maxBurnSteps || fuelLevel cell <= 0.0
         then cell { cellState = Burned, burnSteps = newBurnSteps }
         else cell { burnSteps = newBurnSteps }
    Burned -> cell