module Main (main) where

import Control.Monad (forM_)
import Data.Char (isSpace)
import Data.Foldable (foldl')
import Data.Integer.SAT
  ( Expr (..),
    Prop (..),
    toName,
  )
import System.Directory (createDirectoryIfMissing, listDirectory)
import System.Environment (getArgs)
import System.FilePath (replaceExtension, takeExtension, (</>))

-- Main: two modes
--
-- 1) dimacs2prop-exe file.cnf
--      -> writes file.prop next to it
--
-- 2) dimacs2prop-exe cnfDir propDir
--      -> converts all .cnf in cnfDir to .prop in propDir

main :: IO ()
main = do
  args <- getArgs
  case args of
    [cnfPath] -> do
      putStrLn $ "Converting single file: " ++ cnfPath
      let outPath = replaceExtension cnfPath ".prop"
      convertFile cnfPath outPath
    [cnfDir, propDir] -> do
      putStrLn $ "Converting directory: " ++ cnfDir ++ " -> " ++ propDir
      createDirectoryIfMissing True propDir
      files <- listDirectory cnfDir
      let cnfFiles = [f | f <- files, takeExtension f == ".cnf"]
      forM_ cnfFiles $ \f -> do
        let inPath = cnfDir </> f
            outPath = propDir </> replaceExtension f ".prop"
        putStrLn $ "  " ++ inPath ++ " -> " ++ outPath
        convertFile inPath outPath
    _ ->
      putStrLn $
        "Usage:\n"
          ++ "  dimacs2prop-exe file.cnf\n"
          ++ "  dimacs2prop-exe cnfDir propDir\n"

--------------------------------------------------------------------------------
-- Per-file conversion

convertFile :: FilePath -> FilePath -> IO ()
convertFile inPath outPath = do
  contents <- readFile inPath
  let (numVars, clauses) = parseDIMACS contents
      p = dimacsToProp numVars clauses
      s = renderProp p ++ "\n"
  writeFile outPath s

--------------------------------------------------------------------------------
-- DIMACS parsing

-- Parse DIMACS CNF contents into (number of variables, list of clauses)
-- Each clause is a list of Int literals (e.g. [26,-99,7])
parseDIMACS :: String -> (Int, [[Int]])
parseDIMACS txt =
  let ls = lines txt
      nonComment = filter (\l -> null l || head l /= 'c') ls
      (hdr, restLines) =
        case break (\l -> not (null l) && head l == 'p') nonComment of
          (_, []) -> error "No 'p cnf' header line found"
          (before, pLine : after) -> (pLine, before ++ after)

      -- Option A: total pattern match (no non-exhaustive warning)
      (_cnfStr, nStr, mStr) =
        case words hdr of
          ("p" : cnfStr' : nStr' : mStr' : _) -> (cnfStr', nStr', mStr')
          ws ->
            error $
              "Bad DIMACS header (expected: p cnf <nvars> <nclauses>): "
                ++ unwords ws

      numVars = read nStr :: Int
      _numClauses = read mStr :: Int

      clauseLines =
        takeWhile
          ( \l -> case dropWhile isSpace l of
              [] -> True
              (c : _) -> c /= '%'
          )
          restLines

      toks = concatMap words clauseLines
      clauses = parseClauses toks
   in (numVars, clauses)

parseClauses :: [String] -> [[Int]]
parseClauses = go [] []
  where
    go acc current [] =
      reverse (if null current then acc else reverse current : acc)
    go acc current (t : ts)
      | all isSpace t = go acc current ts
      | otherwise =
          let n = read t :: Int
           in if n == 0
                then go (reverse current : acc) [] ts
                else go acc (n : current) ts

--------------------------------------------------------------------------------
-- DIMACS CNF -> Prop

-- Conjunction helper
conj :: [Prop] -> Prop
conj [] = PTrue
conj [p] = p
conj (p : ps) = p :&& conj ps

-- Build the single Prop encoding:
--  * For each boolean variable x_i: 0 <= Xi <= 1
--  * For each clause: sum(f literal) >= 1
dimacsToProp :: Int -> [[Int]] -> Prop
dimacsToProp numVars clauses =
  let -- Variable domain: Xi ∈ {0,1}
      varProps :: [Prop]
      varProps =
        [ let xi = Var (toName (i - 1))
           in (K 0 :<= xi) :&& (xi :<= K 1)
          | i <- [1 .. numVars]
        ]

      -- Clause constraints
      clauseProps :: [Prop]
      clauseProps = map clauseProp clauses
   in conj (varProps ++ clauseProps)

-- Convert a DIMACS clause [l1,l2,...] to a Prop: sum(f(li)) >= 1
-- Convert a DIMACS clause [l1,l2,...] to a Prop: sum(f(li)) >= 1
clauseProp :: [Int] -> Prop
clauseProp lits =
  let -- DIMACS literal k -> linear 0/1 expression in Xi
      --  k > 0  : x_k
      --  k < 0  : 1 - x_|k|  encoded as 1 + (-1)*x_|k|
      litExpr :: Int -> Expr
      litExpr k
        | k > 0 = Var (toName (k - 1))
        | k < 0 = K 1 :+ Negate (Var (toName (abs k - 1)))
        | otherwise = error "literal 0 should not appear here"

      sumExpr :: Expr
      sumExpr =
        case map litExpr lits of
          [] -> K 0
          (e : es) -> foldl' (:+) e es
   in sumExpr :>= K 1

renderProp :: Prop -> String
renderProp PTrue = "PTrue"
renderProp PFalse = "PFalse"
renderProp (Not p) = "(Not " ++ renderProp p ++ ")"
renderProp (p1 :&& p2) =
  "(" ++ renderProp p1 ++ " :&& " ++ renderProp p2 ++ ")"
renderProp (p1 :|| p2) =
  "(" ++ renderProp p1 ++ " :|| " ++ renderProp p2 ++ ")"
renderProp (e1 :== e2) = "(" ++ show e1 ++ " :== " ++ show e2 ++ ")"
renderProp (e1 :/= e2) = "(" ++ show e1 ++ " :/= " ++ show e2 ++ ")"
renderProp (e1 :< e2) = "(" ++ show e1 ++ " :<  " ++ show e2 ++ ")"
renderProp (e1 :<= e2) = "(" ++ show e1 ++ " :<= " ++ show e2 ++ ")"
renderProp (e1 :> e2) = "(" ++ show e1 ++ " :>  " ++ show e2 ++ ")"
renderProp (e1 :>= e2) = "(" ++ show e1 ++ " :>= " ++ show e2 ++ ")"