--Name: Zack Singerman
--Uni: zs2661
--

import Control.Monad.State
import Control.Monad (guard)
import Control.Applicative (Alternative(..))
import qualified Data.Map as Map
import Data.Map (Map)
import Data.List (maximumBy)
import Data.Function (on)

import Control.Parallel.Strategies (withStrategy, parList, rdeepseq, rpar, rseq, runEval)
import Control.DeepSeq (NFData(..), rnf)




data Point --Where a piece is on the board 
    = P Int --On the board itself, from 1-24
    | Bar --Captured piece which has to come in from off of the board
    | BearOff --Player that can get into the winning spot
    deriving (Eq, Ord, Show)

data Player = White | Black
    deriving (Eq, Ord, Show)
 
------------------------Board representation------------------------------------
--------------------------------------------------------------------------------

type Stack = (Player, Int) --Represents all pieces on a point
                           --Player is which player owns the point
                           --Int is how many pieces are there

type BoardPoints = Map Point Stack --Maps board points to stacks

data Board = Board
    { points  :: BoardPoints    --The map of points on stacks
    , bar     :: Map Player Int --How many pieces each person has captured
    , bearOff :: Map Player Int --How many players each one has borne off
    }
    deriving (Eq, Show)

---------------------Moving-----------------------------------------------------
--------------------------------------------------------------------------------

data CheckerMove = CheckerMove
    { from :: Point
    , to   :: Point
    }
    deriving (Eq, Show)

data Move = Move --The move made
    { movesUsed :: [CheckerMove]
    }
    deriving (Eq, Show)

--NFData instances for parallel evaluation
instance NFData Point where
    rnf (P n)   = rnf n
    rnf Bar     = ()
    rnf BearOff = ()

instance NFData Player where
    rnf White = ()
    rnf Black = ()

instance NFData CheckerMove where
    rnf (CheckerMove from to) = rnf from `seq` rnf to

instance NFData Move where
    rnf (Move movesUsed) = rnf movesUsed

--------------------------Dice--------------------------------------------------
--------------------------------------------------------------------------------

data Dice = Dice Int Int
    deriving (Eq, Show)

diceValues :: Dice -> [Int]
diceValues (Dice a b)
  | a == b    = replicate 4 a --The rule of doubles in Backgammon
  | otherwise = [a, b]

diceOrders :: Dice -> [[Int]] --As Backgammon moves are based on order
diceOrders (Dice a b)
  | a == b    = [replicate 4 a]
  | otherwise = [[a,b],[b,a]]

----------------------The MoveGen monad-----------------------------------------
--------------------------------------------------------------------------------

type MoveGen a = StateT Board [] a --Our own Monad
    --StateT allows for Board to change as moves are made
    --[] allows for multiple outcomes to be explored
    --Illegal moves are not explored (best part of this monad)

---------------Bar Rule---------------------------------------------------------
--------------------------------------------------------------------------------

hasBarPieces :: Board -> Player -> Bool --Does the player have captured pieces?
hasBarPieces board player =
    Map.findWithDefault 0 player (bar board) > 0

---------------------FROM points------------------------------------------------
--------------------------------------------------------------------------------

legalFromPoint :: Player -> MoveGen Point --Finds all legal moving points
legalFromPoint player = do
    currentBoard <- get
    legalFromBar currentBoard <|> legalFromPoints currentBoard
  where
    legalFromBar board =
        guard (hasBarPieces board player) >> pure Bar
    legalFromPoints board =
        lift [ point
             | (point, (owner, n)) <- Map.toList (points board)
             , owner == player
             , n > 0
             ]

----------------Special Moves (From the bar and bearing off)--------------------
--------------------------------------------------------------------------------

entryPoint :: Player -> Int -> Point --Entry point for each player
entryPoint White d = P (25 - d)
entryPoint Black d = P d

homeQuadrant :: Player -> [Point] --Home quadrant for each player
homeQuadrant White = map P [1..6]
homeQuadrant Black = map P [19..24]

canBearOff :: Board -> Player -> Bool --Needed for winning the game
canBearOff board player =
    Map.findWithDefault 0 player (bar board) == 0 --No pieces on the bar
    &&
    all (\(p,(pl,_)) -> pl /= player || p `elem` homeQuadrant player)
        (Map.toList (points board))
        --All pieces are in the home quadrant

------------------TO Points-----------------------------------------------------
--------------------------------------------------------------------------------

isBearOff :: Point -> Bool 
isBearOff BearOff = True
isBearOff _       = False
--To label if a piece can get to the outcome of bearing off


notBlocked :: Board -> Player -> Point -> Bool 
--Are there 2 or more opponent pieces on a location?
notBlocked board movingPlayer destination =
    case Map.lookup destination (points board) of
      Nothing -> --If there is no stack, then the space is open
        True

      Just (owner, numberOfPieces) --Now we use the stack
        | owner == movingPlayer ->
            True

        | numberOfPieces == 1 -> --We can capture
            True

        | otherwise ->
            False

legalDestination :: Player -> Point -> Int -> MoveGen Point
legalDestination player from die = do
    board <- get
    let dest =
          case from of
            Bar -> entryPoint player die
            P pointNumber | player == White -> P (pointNumber - die)
                          | otherwise       -> P (pointNumber + die)
            _ -> error "illegal from-point" --BearOff should not be a fromPoint

    bearOffMove board dest <|> regularMove board dest
  where
    bearOffMove board dest =
        guard (isBearOff dest) >> guard (canBearOff board player) >> pure BearOff
    regularMove board dest =
        guard (notBlocked board player dest) >> pure dest
    

--------------------Applying a move---------------------------------------------
--------------------------------------------------------------------------------

applyCheckerMove :: Board -> Player -> CheckerMove -> Board 
--Returns an updated board
applyCheckerMove board movingPlayer (CheckerMove startPoint endPoint) =
    board
      { points  = updatedPoints
      , bar     = updatedBar
      , bearOff = updatedBearOff
      }
  where
    currentPoints  = points board
    currentBar     = bar board
    currentBearOff = bearOff board

    opponent =
      case movingPlayer of
        White -> Black
        Black -> White

    --Remove pieces from starting point
    pointsAfterRemoval =
      case startPoint of

        Bar -> --Moving piece from the bar
          currentPoints

        P pointNumber -> --Moving from a regular board spot (pointNumber)
          case Map.lookup startPoint currentPoints of
            Just (owner, numberOfPieces)
              | numberOfPieces > 1 ->
                  Map.insert startPoint (owner, numberOfPieces - 1) currentPoints
              | otherwise ->
                  Map.delete startPoint currentPoints

            _ -> --Should not happen, here just in case
              error "applyCheckerMove: invalid start point"

        BearOff -> --Should not happen, here just in case
          error "applyCheckerMove: cannot move from BearOff"

    updatedBarAfterRemoval =
      case startPoint of
        Bar ->
          Map.adjust (subtract 1) movingPlayer currentBar
        _ ->
          currentBar

    --Add pieces to TO Point
    (updatedPoints, updatedBar, updatedBearOff) =
      case endPoint of

        BearOff -> --Removing pieces to win
          ( pointsAfterRemoval
          , updatedBarAfterRemoval
          , Map.insertWith (+) movingPlayer 1 currentBearOff
          )

        P destinationPoint -> --Regular moves
          case Map.lookup endPoint pointsAfterRemoval of

            Nothing -> --Empty space
              ( Map.insert endPoint (movingPlayer, 1) pointsAfterRemoval
              , updatedBarAfterRemoval
              , currentBearOff
              )

            Just (owner, numberOfPieces) --If not free

              | owner == movingPlayer -> --If player owns the space
                  ( Map.insert endPoint (owner, numberOfPieces + 1) pointsAfterRemoval
                  , updatedBarAfterRemoval
                  , currentBearOff
                  )

              | numberOfPieces == 1 -> --Capturing a piece
                  ( Map.insert endPoint (movingPlayer, 1) pointsAfterRemoval
                  , Map.insertWith (+) opponent 1 updatedBarAfterRemoval
                  , currentBearOff
                  )

              | otherwise -> --Should not happen, here just in case
                  error "applyCheckerMove: move to blocked point"

        Bar -> --Should not happen, here just in case 
               --Cannot move a piece to the bar
          error "applyCheckerMove: cannot move to Bar"

------------------Moving one piece----------------------------------------------
--------------------------------------------------------------------------------

singleCheckerMove :: Player -> Int -> MoveGen CheckerMove 
--Actually moves the piece
singleCheckerMove player die = do
    from <- legalFromPoint player
    to   <- legalDestination player from die
    let move = CheckerMove from to
    modify (\board -> applyCheckerMove board player move) 
    --Runs this move into the big system
    pure move


--------------------Full turn---------------------------------------------------
--------------------------------------------------------------------------------

turnMove :: Player -> [Int] -> MoveGen Move --All possible moves based on dice
turnMove player dice =
    Move <$> mapM (singleCheckerMove player) dice -- <$> is the same as fmap
                                                   -- Looks cooler
    --This braches all the different possibilities

generateMoves :: Board -> Player -> Dice -> [Move] 
--Generate all legal moves from a position
generateMoves board player dice =
    concat
      [ evalStateT (turnMove player ds) board --Returns a move
      | ds <- diceOrders dice --The possible dice orders for moves
      ]

--------------------Game Over Check---------------------------------------------
--------------------------------------------------------------------------------

isGameOver :: Board -> Maybe Player
isGameOver board --Checks if a player has borne off all pieces
  | Map.findWithDefault 0 White (bearOff board) >= 15 = Just White
  | Map.findWithDefault 0 Black (bearOff board) >= 15 = Just Black
  | otherwise = Nothing

countBorneOff :: Player -> Board -> Int --Good for heuristics later
countBorneOff player board =
    Map.findWithDefault 0 player (bearOff board)

countOnBar :: Player -> Board -> Int --Good for heuristics later
countOnBar player board =
    Map.findWithDefault 0 player (bar board)

--------------------Board Evaluator---------------------------------------------
--------------------------------------------------------------------------------

evaluateBoard :: Player -> Board -> Int --Heuristics for expectiminimax
evaluateBoard player board =
    let
        borneOffScore = 10 * countBorneOff player board 
        --10 points for borne off pieces

        blots = length --A blot is a single & vulnerable piece
          [ ()
          | (point, (ownerOfPoint, numCheckers)) <- Map.toList (points board)
          , ownerOfPoint == player
          , numCheckers == 1
          ]
        blotPenalty = -10 * blots

        protectedStacks = length 
        --Looks for how many slots of the board that player owns
          [ ()
          | (point, (ownerOfPoint, numCheckers)) <- Map.toList (points board)
          , ownerOfPoint == player
          , numCheckers >= 2
          ]
        protectedBonus = 5 * protectedStacks
    in
        borneOffScore + blotPenalty + protectedBonus

distanceToBearOff :: Player -> Point -> Int
distanceToBearOff player (P pointNumber) =
    case player of
      White -> pointNumber - 1        --White moves from 24 to 1
      Black -> 24 - pointNumber       --Black moves from 1 to 24
distanceToBearOff _ Bar = 25          --A piece on the bar is far away
distanceToBearOff _ BearOff = 0       --Already borne off

--------------------Next Player Helper------------------------------------------
--------------------------------------------------------------------------------

nextPlayer :: Player -> Player
nextPlayer White = Black
nextPlayer Black = White

--------------------Apply Full Move---------------------------------------------
--------------------------------------------------------------------------------

applyMoveSequence :: Board -> Player -> Move -> Board
--Applying a full move, sequence of pieces moving, to the board
applyMoveSequence board player (Move checkerMoves) =
    foldl
      (\currentBoard checkerMove ->
          applyCheckerMove currentBoard player checkerMove)
      board
      checkerMoves

--------------------Dice Probabilities------------------------------------------
--------------------------------------------------------------------------------

diceOutcomes :: [(Dice, Double)]
--All possible dice outcome
diceOutcomes =
    [ (Dice a b, probability a b)
    | a <- [1..6]
    , b <- [a..6]
    ]
  where
    probability a b --Probability of each dice outcome
      | a == b    = 1.0 / 36.0
      | otherwise = 2.0 / 36.0

--------------------Expectiminimax with Alpha-Beta Pruning----------------------
--------------------------------------------------------------------------------

searchDepth :: Int
searchDepth = 3
--4 plies: Max, Min, Max, Min

expectiminimaxAB
  :: Int        --Search depth
  -> Player     --Max player (root)
  -> Player     --Current player
  -> Double     --Alpha
  -> Double     --Beta
  -> Board
  -> Double
expectiminimaxAB depth maximizingPlayer currentPlayer alpha beta board

  | depth == 0 = --Stop searching
      fromIntegral (evaluateBoard maximizingPlayer board)

  | Just winner <- isGameOver board = --Game over
      if winner == maximizingPlayer then 1e9 else -1e9
      --1e9 to represent the largest or smaller number possible
      --Essentially represents winning and losing

  | otherwise = --Chance Node, No pruning
      sum
        [ probability * bestResponse dice
        | (dice, probability) <- diceOutcomes
        ]

  where
    bestResponse dice
      | null moves = fromIntegral (evaluateBoard maximizingPlayer board)
      | currentPlayer == maximizingPlayer = maxValue moves alpha beta
      | otherwise = minValue moves alpha beta
      where
        moves = generateMoves board currentPlayer dice

    maxValue :: [Move] -> Double -> Double -> Double --Max Node
    maxValue [] currentAlpha _ = currentAlpha
    maxValue (move:moves) currentAlpha currentBeta =
        let
            score =
              expectiminimaxAB
                (depth - 1)
                maximizingPlayer
                (nextPlayer currentPlayer)
                currentAlpha
                currentBeta
                (applyMoveSequence board currentPlayer move)

            newAlpha = max currentAlpha score
        in
            if newAlpha >= currentBeta
               then newAlpha      --Beta cutoff
               else maxValue moves newAlpha currentBeta

    
    minValue :: [Move] -> Double -> Double -> Double --Min Node
    minValue [] _ currentBeta = currentBeta
    minValue (move:moves) currentAlpha currentBeta =
        let
            score =
              expectiminimaxAB
                (depth - 1)
                maximizingPlayer
                (nextPlayer currentPlayer)
                currentAlpha
                currentBeta
                (applyMoveSequence board currentPlayer move)

            newBeta = min currentBeta score
        in
            if currentAlpha >= newBeta
               then newBeta --Alpha cutoff
               else minValue moves currentAlpha newBeta

--------------------Choose Best Move Based on Dice (With Parallel)--------------
--------------------------------------------------------------------------------

parMapEval :: (a -> b) -> [a] -> [b]
parMapEval f xs = runEval $ do
  ys <- mapM (rpar . f) xs --Spark the parallel
  mapM rseq ys --Force results, hurts runtime but gets stuff done


chooseBestMoveForDiceAB --Actually chooses the best move
  :: Int        --Search depth
  -> Player     --Player to move
  -> Board
  -> Dice
  -> Maybe (Move, Double)
chooseBestMoveForDiceAB depth player board dice =
    case scoredMoves of
      [] -> Nothing
      _  -> Just (maximumBy (compare `on` snd) scoredMoves)
  where
    moves :: [Move]
    moves = generateMoves board player dice

    scoredMoves :: [(Move, Double)]
    scoredMoves =
      withStrategy (parList rdeepseq) $
        map scoreMove (moves :: [Move])


    scoreMove :: Move -> (Move, Double)
    scoreMove move =
      let !score =
            expectiminimaxAB
              (depth - 1)
              player
              (nextPlayer player)
              (-1e9)
              (1e9)
              (applyMoveSequence board player move)
      in (move, score)

--------------------Runs Choose-Best-Move and Prints what it is-----------------
--------------------------------------------------------------------------------

analyzePosition --Prints the best move and its expected value.
  :: Board
  -> Player
  -> Dice
  -> IO ()
analyzePosition board player dice =
    case chooseBestMoveForDiceAB searchDepth player board dice of
      Nothing ->
        putStrLn "No legal moves."

      Just (bestMove, score) -> do
        putStrLn "========================================"
        putStrLn ("Player to move: " ++ show player)
        putStrLn ("Dice roll: " ++ show dice)
        putStrLn "Best move:"
        print bestMove
        putStrLn ("Expected score: " ++ show score)
        putStrLn "========================================"

--------------------Input your board here to see best move----------------------
--------------------------------------------------------------------------------

testBoard :: Board 
--Input your own board configuration, examples below
testBoard =
  Board
    { points =
        Map.fromList
          [ (P 1,   (White, 0))
          , (P 2,   (White, 0))
          , (P 3,   (White, 0))
          , (P 4,   (White, 0))
          , (P 5,   (White, 0))
          , (P 6,   (White, 0))
          , (P 7,   (White, 0))
          , (P 8,   (White, 0))
          , (P 9,   (White, 0))
          , (P 10,  (White, 0))
          , (P 11,  (White, 0))
          , (P 12,  (White, 0))
          , (P 13,  (Black, 0))
          , (P 14,  (Black, 0))
          , (P 15,  (Black, 0))
          , (P 16,  (Black, 0))
          , (P 17,  (Black, 0))
          , (P 18,  (Black, 0))
          , (P 19,  (Black, 0))
          , (P 20,  (Black, 0))
          , (P 21,  (Black, 0))
          , (P 22,  (Black, 0))
          , (P 23,  (Black, 0))
          , (P 24,  (Black, 0))
          ]
    , bar = Map.fromList [(White, 0), (Black, 0)]
    , bearOff = Map.fromList [(White, 0), (Black, 0)]
    }

---------Four Possible Testing boards:

----------------Midgame, both players have some pieces on bar
midgameBoard :: Board
midgameBoard = Board
  { points = Map.fromList
      [ (P 3,  (White, 2))
      , (P 8,  (White, 3))
      , (P 10, (White, 3))
      , (P 16, (White, 3))
      , (P 21, (White, 2))
      , (P 5,  (Black, 3))
      , (P 12, (Black, 3))
      , (P 19, (Black, 2))
      , (P 23, (Black, 2))
      , (P 24, (Black, 3))
      ]
  , bar = Map.fromList [(White,1), (Black,2)]
  , bearOff = Map.fromList [(White,0), (Black,0)]
  }

----------------Endgame scenario for White
endgameWhite :: Board
endgameWhite = Board
  { points = Map.fromList
      [ (P 1, (White, 3))
      , (P 2, (White, 3))
      , (P 4, (White, 3))
      , (P 6, (Black, 3))
      , (P 12, (Black, 4))
      , (P 18, (Black, 2))
      , (P 19, (Black, 2))
      , (P 22, (Black, 3))
      ]
  , bar = Map.fromList [(White,0), (Black,1)]
  , bearOff = Map.fromList [(White,6), (Black,0)]
  }

----------------Critical hit scenario
criticalHitBoard :: Board
criticalHitBoard = Board
  { points = Map.fromList
      [ (P 5,  (White, 4))  
      , (P 7,  (White, 1))
      , (P 11, (White, 3))
      , (P 16, (White, 3))
      , (P 22, (White, 1))
      , (P 3,  (Black, 1))  
      , (P 6,  (Black, 4))
      , (P 12, (Black, 1))
      , (P 19, (Black, 5))
      , (P 24, (Black, 3))  
      ]
  , bar = Map.fromList [(White,1), (Black,1)]   
  , bearOff = Map.fromList [(White,2), (Black,0)]
  }

----------------All pieces ready to bear off
bearOffReadyBoard :: Board
bearOffReadyBoard = Board
  { points = Map.fromList
      [ (P 1,  (White, 3))
      , (P 2,  (White, 1))
      , (P 4,  (White, 4))
      , (P 6,  (White, 3))
      , (P 19, (Black, 3))
      , (P 20, (Black, 3))
      , (P 24, (Black, 2))
      ]
  , bar = Map.fromList [(White,0), (Black,0)]
  , bearOff = Map.fromList [(White,4), (Black,7)]
  }

startingBoard :: Board --Starting board for every game of backgammon
startingBoard =
  Board
    { points =
        Map.fromList
          [ --White Pieces
            (P 24, (White, 2))   
          , (P 13, (White, 5))   
          , (P 8,  (White, 3))   
          , (P 6,  (White, 5))   

            --Black Pieces
          , (P 1,  (Black, 2))   
          , (P 12, (Black, 5))   
          , (P 17, (Black, 3))   
          , (P 19, (Black, 5))  
          ]
    , bar =
        Map.fromList
          [ (White, 0)
          , (Black, 0)
          ]
    , bearOff =
        Map.fromList
          [ (White, 0)
          , (Black, 0)
          ]
    }


exampleDice :: Dice --Input whatever roll you got
exampleDice = Dice 3 5

--------------------Main--------------------------------------------------------
--------------------------------------------------------------------------------

main :: IO ()
main =
  analyzePosition endgameWhite White exampleDice


