{-# LANGUAGE ScopedTypeVariables #-}

module ReadInstances
  ( KnapsackInstance(..)
  , readAllInstances
  ) where

import System.Directory
import System.FilePath
import Control.Monad
import Text.Read (readMaybe)

------------------------------------------------------------
-- Represents one knapsack problem instance
------------------------------------------------------------

data KnapsackInstance = KnapsackInstance
  { inst_name :: FilePath
  , n_items   :: Int
  , items     :: [(Double, Double)]   -- (value, weight)
  , capacity  :: Double
  } deriving (Show)

------------------------------------------------------------
-- Read one .in file
------------------------------------------------------------

readInstanceFile :: FilePath -> IO KnapsackInstance
readInstanceFile file_path = do
  contents <- readFile file_path
  let ls = lines contents

  case ls of
    [] ->
      error $ "[ERROR] empty file: " ++ file_path

    (first_line : rest_lines) ->
      case words first_line of
        [n_str, cap_str] ->
          case ( readMaybe n_str  :: Maybe Int
               , readMaybe cap_str :: Maybe Double
               ) of
            (Just n, Just cap) ->
              if length rest_lines < n
                then error $
                  "[ERROR] not enough item lines in " ++ file_path
                else do
                  let item_lines = take n rest_lines

                  parsed_items <- forM item_lines $ \ln ->
                    case map readMaybe (words ln) of
                      [Just value, Just weight] ->
                        pure (value, weight)
                      _ ->
                        error $
                          "[ERROR] malformed item line: " ++ ln

                  pure KnapsackInstance
                    { inst_name = takeBaseName file_path
                    , n_items   = n
                    , items     = parsed_items
                    , capacity  = cap
                    }

            _ ->
              error $
                "[ERROR] first line must be: <n_items> <capacity> in "
                ++ file_path

        _ ->
          error $
            "[ERROR] first line must be: <n_items> <capacity> in "
            ++ file_path

------------------------------------------------------------
-- Read all instances under input/
-- Expects files like: input/*.in
------------------------------------------------------------

readAllInstances :: FilePath -> IO [KnapsackInstance]
readAllInstances input_root = do
  exists <- doesDirectoryExist input_root
  unless exists $
    error $ "[ERROR] input root not found: " ++ input_root

  files <- listDirectory input_root

  let in_files =
        [ input_root </> f
        | f <- files
        , takeExtension f == ".in"
        ]

  forM in_files readInstanceFile
