#!/usr/bin/env bash
set -euo pipefail

# ----------------------------
# Usage:
#   bash scripts/run_scaling.sh <puzzle_file> <depth> [order] [repeats]
#
# Examples:
#   bash scripts/run_scaling.sh puzzles/curated/7_easy.txt 2 ordered 5
#   bash scripts/run_scaling.sh puzzles/curated/7_easy.txt 3 random  5
# ----------------------------

PUZZLE="${1:-}"
DEPTH="${2:-}"
ORDER="${3:-ordered}"
REPEATS="${4:-5}"

if [[ -z "$PUZZLE" || -z "$DEPTH" ]]; then
  echo "Usage: bash scripts/run_scaling.sh <puzzle_file> <depth> [order] [repeats]"
  exit 1
fi

if [[ ! -f "$PUZZLE" ]]; then
  echo "Error: puzzle file not found: $PUZZLE"
  exit 1
fi

RAW_DIR="results/raw/scaling"
CSV_DIR="results/csv"
mkdir -p "$RAW_DIR" "$CSV_DIR"

CSV_FILE="$CSV_DIR/scaling.csv"
if [[ ! -f "$CSV_FILE" ]]; then
  echo "timestamp,puzzle,mode,depth,order,cores,run_id,runtime_s,solutions" > "$CSV_FILE"
fi

timestamp_now() {
  date +"%Y-%m-%dT%H:%M:%S"
}

extract_runtime() {
  local file="$1"
  # Runtime (first solution): 3.174501 seconds
  grep -E "Runtime \(first solution\):" "$file" | head -n 1 \
    | awk -F': ' '{print $2}' | awk '{print $1}' || true
}

extract_solutions() {
  local file="$1"
  grep -E "Number of solutions:" "$file" | head -n 1 \
    | awk -F': ' '{print $2}' | awk '{print $1}' || true
}

run_one() {
  local cores="$1"
  local run_id="$2"
  local mode="par"

  local base
  base="$(basename "$PUZZLE")"
  local raw_file="${RAW_DIR}/${base%.txt}_${mode}_d${DEPTH}_${ORDER}_N${cores}_run${run_id}.txt"

  # Run and save stdout
  stack exec futoshiki -- "$PUZZLE" par "$DEPTH" "$ORDER" +RTS -N"$cores" -RTS \
    | tee "$raw_file" >/dev/null

  local rt sol ts
  rt="$(extract_runtime "$raw_file")"
  sol="$(extract_solutions "$raw_file")"
  ts="$(timestamp_now)"

  if [[ -z "$rt" ]]; then
    echo "Warning: failed to parse runtime from $raw_file"
    rt=""
  fi

  echo "${ts},${PUZZLE},${mode},${DEPTH},${ORDER},${cores},${run_id},${rt},${sol}" >> "$CSV_FILE"
  echo "[DONE] N=$cores run=$run_id runtime=${rt}s"
}

echo "=== Scaling Experiment ==="
echo "Puzzle:   $PUZZLE"
echo "Depth:    $DEPTH"
echo "Order:    $ORDER"
echo "Repeats:  $REPEATS"
echo "Outputs:  $RAW_DIR and $CSV_FILE"
echo

# Warmup (not recorded)
echo "[Warmup] par depth=$DEPTH order=$ORDER (not recorded)"
stack exec futoshiki -- "$PUZZLE" par "$DEPTH" "$ORDER" +RTS -N1 -RTS >/dev/null || true
echo

CORES_LIST=(1 2 4 8)

for c in "${CORES_LIST[@]}"; do
  echo "=== Running: par depth=$DEPTH $ORDER (N=$c) x $REPEATS ==="
  for i in $(seq 1 "$REPEATS"); do
    run_one "$c" "$i"
  done
  echo
done

echo "All scaling runs saved."
echo
