#!/usr/bin/env bash
set -euo pipefail

# ----------------------------
# Parallel ordered vs random experiment
#
# Usage:
#   bash scripts/run_par_order_compare.sh <puzzle_file> <depth> [cores] [repeats]
#
# Example:
#   bash scripts/run_par_order_compare.sh puzzles/curated/7_easy.txt 3 8 10
# ----------------------------

PUZZLE="${1:-}"
DEPTH="${2:-}"
CORES="${3:-8}"
REPEATS="${4:-10}"

if [[ -z "$PUZZLE" || -z "$DEPTH" ]]; then
  echo "Usage: bash scripts/run_par_order_compare.sh <puzzle_file> <depth> [cores] [repeats]"
  exit 1
fi

if [[ ! -f "$PUZZLE" ]]; then
  echo "Error: puzzle file not found: $PUZZLE"
  exit 1
fi

RAW_DIR="results/raw/par_order_compare"
CSV_DIR="results/csv"
mkdir -p "$RAW_DIR" "$CSV_DIR"

CSV_FILE="$CSV_DIR/par_order_compare.csv"
if [[ ! -f "$CSV_FILE" ]]; then
  echo "timestamp,puzzle,mode,depth,order,cores,run_id,runtime_s,solutions" > "$CSV_FILE"
fi

timestamp_now() { date +"%Y-%m-%dT%H:%M:%S"; }

extract_runtime() {
  local file="$1"
  grep -E "Runtime \(first solution\):" "$file" | head -n 1 \
    | awk -F': ' '{print $2}' | awk '{print $1}' || true
}

extract_solutions() {
  local file="$1"
  grep -E "Number of solutions:" "$file" | head -n 1 \
    | awk -F': ' '{print $2}' | awk '{print $1}' || true
}

run_one() {
  local order="$1"
  local run_id="$2"
  local mode="par"
  local base
  base="$(basename "$PUZZLE")"
  local raw_file="${RAW_DIR}/${base%.txt}_${mode}_d${DEPTH}_${order}_N${CORES}_run${run_id}.txt"

  stack exec futoshiki -- "$PUZZLE" par "$DEPTH" "$order" +RTS -N"$CORES" -RTS \
    | tee "$raw_file" >/dev/null

  local rt sol ts
  rt="$(extract_runtime "$raw_file")"
  sol="$(extract_solutions "$raw_file")"
  ts="$(timestamp_now)"

  echo "${ts},${PUZZLE},${mode},${DEPTH},${order},${CORES},${run_id},${rt},${sol}" >> "$CSV_FILE"
  echo "[DONE] order=$order run=$run_id runtime=${rt}s"
}

echo "=== Par Ordered vs Random ==="
echo "Puzzle:   $PUZZLE"
echo "Depth:    $DEPTH"
echo "Cores:    $CORES"
echo "Repeats:  $REPEATS"
echo "Outputs:  $RAW_DIR and $CSV_FILE"
echo

# Warmup (not recorded)
echo "[Warmup] par depth=$DEPTH ordered N=$CORES (not recorded)"
stack exec futoshiki -- "$PUZZLE" par "$DEPTH" ordered +RTS -N"$CORES" -RTS >/dev/null || true
echo

for order in ordered random; do
  echo "=== Running: par depth=$DEPTH $order (N=$CORES) x $REPEATS ==="
  for i in $(seq 1 "$REPEATS"); do
    run_one "$order" "$i"
  done
  echo
done

echo "All runs saved."
echo "CSV: $CSV_FILE"
