#!/usr/bin/env bash
set -euo pipefail

# ----------------------------
# Final matrix: depth × order × cores
#
# Usage:
#   bash scripts/run_final_matrix.sh <puzzle_file> [min_depth] [max_depth] [repeats]
#
# Examples:
#   bash scripts/run_final_matrix.sh puzzles/curated/7_easy.txt 0 6 3
#   bash scripts/run_final_matrix.sh puzzles/curated/7_easy.txt 0 4 5
# ----------------------------

PUZZLE="${1:-}"
MIN_DEPTH="${2:-0}"
MAX_DEPTH="${3:-6}"
REPEATS="${4:-3}"

if [[ -z "$PUZZLE" ]]; then
  echo "Usage: bash scripts/run_final_matrix.sh <puzzle_file> [min_depth] [max_depth] [repeats]"
  exit 1
fi

if [[ ! -f "$PUZZLE" ]]; then
  echo "Error: puzzle file not found: $PUZZLE"
  exit 1
fi

RAW_DIR="results/raw/final_matrix"
CSV_DIR="results/csv"
mkdir -p "$RAW_DIR" "$CSV_DIR"

# Write to a unique CSV so you don't mix with older runs
BASE="$(basename "$PUZZLE")"
CSV_FILE="$CSV_DIR/final_matrix_${BASE%.txt}_d${MIN_DEPTH}-${MAX_DEPTH}_r${REPEATS}.csv"

echo "timestamp,puzzle,mode,depth,order,cores,run_id,runtime_s,solutions" > "$CSV_FILE"

timestamp_now() { date +"%Y-%m-%dT%H:%M:%S"; }

extract_runtime() {
  local file="$1"
  grep -E "Runtime \(first solution\):" "$file" | head -n 1 \
    | awk -F': ' '{print $2}' | awk '{print $1}' || true
}

extract_solutions() {
  local file="$1"
  grep -E "Number of solutions:" "$file" | head -n 1 \
    | awk -F': ' '{print $2}' | awk '{print $1}' || true
}

run_one() {
  local depth="$1"
  local order="$2"
  local cores="$3"
  local run_id="$4"
  local mode="par"

  local raw_file="${RAW_DIR}/${BASE%.txt}_${mode}_d${depth}_${order}_N${cores}_run${run_id}.txt"

  stack exec futoshiki -- "$PUZZLE" par "$depth" "$order" +RTS -N"$cores" -RTS \
    | tee "$raw_file" >/dev/null

  local rt sol ts
  rt="$(extract_runtime "$raw_file")"
  sol="$(extract_solutions "$raw_file")"
  ts="$(timestamp_now)"

  echo "${ts},${PUZZLE},${mode},${depth},${order},${cores},${run_id},${rt},${sol}" >> "$CSV_FILE"
  echo "[DONE] d=$depth order=$order N=$cores run=$run_id rt=${rt}s"
}

echo "=== Final Matrix Experiment ==="
echo "Puzzle:   $PUZZLE"
echo "Depth:    ${MIN_DEPTH}..${MAX_DEPTH}"
echo "Repeats:  $REPEATS"
echo "Raw dir:  $RAW_DIR"
echo "CSV:      $CSV_FILE"
echo

# Warmup (not recorded)
echo "[Warmup] par depth=3 ordered N=4 (not recorded)"
stack exec futoshiki -- "$PUZZLE" par 3 ordered +RTS -N4 -RTS >/dev/null || true
echo

CORES_LIST=(1 2 4 8)
ORDERS=(ordered random)

for d in $(seq "$MIN_DEPTH" "$MAX_DEPTH"); do
  for o in "${ORDERS[@]}"; do
    for n in "${CORES_LIST[@]}"; do
      echo "=== par d=$d $o N=$n x $REPEATS ==="
      for i in $(seq 1 "$REPEATS"); do
        run_one "$d" "$o" "$n" "$i"
      done
      echo
    done
  done
done

echo "All final-matrix runs saved."
echo "CSV: $CSV_FILE"
