#!/usr/bin/env bash
set -euo pipefail

# ----------------------------
# Usage:
#   bash scripts/run_depth_sweep.sh <puzzle_file> [order] [cores] [repeats] [max_depth]
#
# Example:
#   bash scripts/run_depth_sweep.sh puzzles/curated/7_easy.txt ordered 4 5 4
# ----------------------------

PUZZLE="${1:-}"
ORDER="${2:-ordered}"
CORES="${3:-4}"
REPEATS="${4:-5}"
MAX_DEPTH="${5:-4}"

if [[ -z "$PUZZLE" ]]; then
  echo "Usage: bash scripts/run_depth_sweep.sh <puzzle_file> [order] [cores] [repeats] [max_depth]"
  exit 1
fi

if [[ ! -f "$PUZZLE" ]]; then
  echo "Error: puzzle file not found: $PUZZLE"
  exit 1
fi

RAW_DIR="results/raw/depth_sweep"
CSV_DIR="results/csv"
mkdir -p "$RAW_DIR" "$CSV_DIR"

CSV_FILE="$CSV_DIR/depth_sweep.csv"
if [[ ! -f "$CSV_FILE" ]]; then
  echo "timestamp,puzzle,mode,depth,order,cores,run_id,runtime_s,solutions" > "$CSV_FILE"
fi

timestamp_now() { date +"%Y-%m-%dT%H:%M:%S"; }

extract_runtime() {
  local file="$1"
  grep -E "Runtime \(first solution\):" "$file" | head -n 1 \
    | awk -F': ' '{print $2}' | awk '{print $1}' || true
}

extract_solutions() {
  local file="$1"
  grep -E "Number of solutions:" "$file" | head -n 1 \
    | awk -F': ' '{print $2}' | awk '{print $1}' || true
}

run_one() {
  local depth="$1"
  local run_id="$2"
  local mode="par"
  local base
  base="$(basename "$PUZZLE")"
  local raw_file="${RAW_DIR}/${base%.txt}_${mode}_d${depth}_${ORDER}_N${CORES}_run${run_id}.txt"

  stack exec futoshiki -- "$PUZZLE" par "$depth" "$ORDER" +RTS -N"$CORES" -RTS \
    | tee "$raw_file" >/dev/null

  local rt sol ts
  rt="$(extract_runtime "$raw_file")"
  sol="$(extract_solutions "$raw_file")"
  ts="$(timestamp_now)"

  echo "${ts},${PUZZLE},${mode},${depth},${ORDER},${CORES},${run_id},${rt},${sol}" >> "$CSV_FILE"
  echo "[DONE] depth=$depth run=$run_id runtime=${rt}s"
}

echo "=== Depth Sweep Experiment ==="
echo "Puzzle:    $PUZZLE"
echo "Order:     $ORDER"
echo "Cores:     $CORES"
echo "Repeats:   $REPEATS"
echo "Max depth: $MAX_DEPTH"
echo "Outputs:   $RAW_DIR and $CSV_FILE"
echo

# Warmup (not recorded)
echo "[Warmup] par depth=1 order=$ORDER N=$CORES (not recorded)"
stack exec futoshiki -- "$PUZZLE" par 1 "$ORDER" +RTS -N"$CORES" -RTS >/dev/null || true
echo

for d in $(seq 0 "$MAX_DEPTH"); do
  echo "=== Running: par depth=$d $ORDER (N=$CORES) x $REPEATS ==="
  for i in $(seq 1 "$REPEATS"); do
    run_one "$d" "$i"
  done
  echo
done

echo "All depth sweep runs saved."
echo "CSV: $CSV_FILE"
