#!/usr/bin/env bash
set -euo pipefail

# ----------------------------
# Usage:
#   bash scripts/run_baseline.sh <puzzle_file> [repeats]
# Example:
#   bash scripts/run_baseline.sh puzzles/curated/7_easy.txt 10
# ----------------------------

PUZZLE="${1:-}"
REPEATS="${2:-10}"

if [[ -z "$PUZZLE" ]]; then
  echo "Usage: bash scripts/run_baseline.sh <puzzle_file> [repeats]"
  exit 1
fi

if [[ ! -f "$PUZZLE" ]]; then
  echo "Error: puzzle file not found: $PUZZLE"
  exit 1
fi

# Output dirs
RAW_DIR="results/raw/baseline"
CSV_DIR="results/csv"
mkdir -p "$RAW_DIR" "$CSV_DIR"

CSV_FILE="$CSV_DIR/baseline.csv"

# CSV header (create if not exists)
if [[ ! -f "$CSV_FILE" ]]; then
  echo "timestamp,puzzle,mode,order,cores,run_id,runtime_s,solutions" > "$CSV_FILE"
fi

timestamp_now() {
  # ISO8601, local time
  date +"%Y-%m-%dT%H:%M:%S"
}

# Extract runtime number from stdout file
extract_runtime() {
  local file="$1"
  # line example: Runtime (first solution): 3.174501 seconds
  local rt
  rt="$(grep -E "Runtime \(first solution\):" "$file" | head -n 1 | awk -F': ' '{print $2}' | awk '{print $1}' || true)"
  echo "$rt"
}

# Extract solutions count if present (some modes print it)
extract_solutions() {
  local file="$1"
  local sol
  sol="$(grep -E "Number of solutions:" "$file" | head -n 1 | awk -F': ' '{print $2}' | awk '{print $1}' || true)"
  echo "$sol"
}

run_one() {
  local order="$1"
  local run_id="$2"
  local cores="1"
  local mode="seq"

  local base
  base="$(basename "$PUZZLE")"
  local raw_file="${RAW_DIR}/${base%.txt}_${mode}_${order}_N${cores}_run${run_id}.txt"

  # Run and save stdout
  stack exec futoshiki -- "$PUZZLE" seq "$order" +RTS -N1 -RTS | tee "$raw_file" >/dev/null

  local rt sol ts
  rt="$(extract_runtime "$raw_file")"
  sol="$(extract_solutions "$raw_file")"
  ts="$(timestamp_now)"

  # Basic sanity: runtime must exist
  if [[ -z "$rt" ]]; then
    echo "Warning: failed to parse runtime from $raw_file"
    rt=""
  fi

  echo "${ts},${PUZZLE},${mode},${order},${cores},${run_id},${rt},${sol}" >> "$CSV_FILE"
  echo "[DONE] order=$order run=$run_id runtime=${rt}s"
}

echo "=== Baseline ==="
echo "Puzzle:   $PUZZLE"
echo "Repeats:  $REPEATS"
echo "Outputs:  $RAW_DIR and $CSV_FILE"
echo

# Optional warmup (not recorded) to reduce first-run noise
echo "[Warmup] seq ordered (not recorded)"
stack exec futoshiki -- "$PUZZLE" seq ordered +RTS -N1 -RTS >/dev/null || true
echo

for order in ordered random; do
  echo "=== Running: seq $order (N1) x $REPEATS ==="
  for i in $(seq 1 "$REPEATS"); do
    run_one "$order" "$i"
  done
  echo
done

echo "All baseline runs saved."
echo "CSV: $CSV_FILE"
