{-# OPTIONS_GHC -Wall #-}

{-
Name:
Uni:

Futoshiki Solver (sequential + parallel)
-}

------------------------------------------------------------
-- Imports
------------------------------------------------------------

import qualified Data.Map.Strict as M
import           Data.Map.Strict (Map)
import           Data.List       (minimumBy, sortBy)
import           Data.Ord        (comparing)
import           System.Environment (getArgs, getProgName)
import           System.Exit     (die)
import           Data.Char       (isDigit)
import           Text.Read       (readMaybe)

import           Control.Parallel.Strategies (using, parList, rseq)
import           Data.Maybe (listToMaybe, catMaybes)


import           Data.Time      (getCurrentTime, diffUTCTime)

------------------------------------------------------------
-- Core Types
------------------------------------------------------------

-- Position in the puzzle (r, c)
type Pos   = (Int, Int)

-- Value in the grid
type Value = Int

-- Inequality < and >
data Rel = Lt Pos Pos
          | Gt Pos Pos
          deriving (Show, Eq)

-- Board: Map, Pos is key, Value is value
type Board = Map Pos Value

-- Whole puzzle
data Puzzle = Puzzle Int Board [Rel]
              deriving (Show)

size :: Puzzle -> Int
size (Puzzle n _ _) = n

givens :: Puzzle -> Board
givens (Puzzle _ g _) = g

relations :: Puzzle -> [Rel]
relations (Puzzle _ _ rs) = rs


-- Two Modes for selecting value for a grid
-- Ordered: from 1 to n in order
-- Random : randomly select from 1 to n
data ValueOrder = Ordered
                | Random
                deriving (Show, Eq)

-- Run Modes
--   - RunSeq order          sequential mode, with selecting value in order
--   - RunPar depth order    parallel mode, with selecting value in order
data RunMode = RunSeq ValueOrder
              | RunPar Int ValueOrder
              deriving (Show, Eq)

------------------------------------------------------------
-- NFData instances for parallel evaluation
------------------------------------------------------------
-- instance NFData Rel where
--   rnf (Lt p1 p2) = rnf p1 `seq` rnf p2
--   rnf (Gt p1 p2) = rnf p1 `seq` rnf p2

------------------------------------------------------------
-- Row/column helpers
------------------------------------------------------------

-- Get all values existed in r-th row
rowValues :: Int -> Board -> Int -> [Value]
rowValues n board r = [ v | c <- [0 .. n-1], Just v <- [M.lookup (r,c) board]]

-- Get all values existed in c-th column
colValues :: Int -> Board -> Int -> [Value]
colValues n board c = [ v | r <- [0 .. n-1], Just v <- [M.lookup (r,c) board]]



firstJust :: [Maybe a] -> Maybe a
firstJust xs =
  case catMaybes xs of
    []    -> Nothing
    (y:_) -> Just y
------------------------------------------------------------
-- Relation checking
------------------------------------------------------------

-- Check if the inequality constraints is satisfied
okOneRel :: Board -> Rel -> Bool
okOneRel b (Lt p1 p2) =
  case (M.lookup p1 b, M.lookup p2 b) of
    (Just v1, Just v2) -> v1 < v2
    _                  -> True
okOneRel b (Gt p1 p2) =
  case (M.lookup p1 b, M.lookup p2 b) of
    (Just v1, Just v2) -> v1 > v2
    _                  -> True

-- Check if current value satisfies all constraints
respectsRelations :: [Rel] -> Board -> Pos -> Value -> Bool
respectsRelations rels board pos v =
  let b' = M.insert pos v board
  in all (okOneRel b') rels

------------------------------------------------------------
-- MRV heuristic (Minimum Remaining Values)
------------------------------------------------------------

-- Get all candidates for a grid
candidatesFor :: Puzzle -> Board -> Pos -> [Value]
candidatesFor p board (r,c) = [ v | v <- [1 .. size p], 
                                v `notElem` rowValues (size p) board r, 
                                v `notElem` colValues (size p) board c, 
                                respectsRelations (relations p) board (r,c) v]

-- Find all unfilled positions
unfilledPositions :: Puzzle -> Board -> [Pos]
unfilledPositions p board = [ (r,c) | r <- [0 .. size p - 1], 
                                      c <- [0 .. size p - 1],
                            M.notMember (r,c) board]

-- Choose next unfilled grid
-- Get all candidates for all unfilled grid
-- Choose the grid with the least candidate
-- Return that candidate
chooseNext :: Puzzle -> Board -> Maybe (Pos, [Value])
chooseNext p board =
  case unfilledPositions p board of
    [] -> Nothing  -- No unfilled positions -> solved
    ps ->
      let cs = [ (pos, candidatesFor p board pos) | pos <- ps ]
      in Just (pickMin cs)
  where
    pickMin :: [(Pos, [Value])] -> (Pos, [Value])
    pickMin [] = error "pickMin: empty list"
    pickMin [x] = x
    pickMin (x : y : rest) =
      let (_, vs1) = x
          (_, vs2) = y
      in if length vs1 <= length vs2
           then pickMin (x : rest)
           else pickMin (y : rest)

------------------------------------------------------------
-- Value order: ordered / random
------------------------------------------------------------

orderValues :: ValueOrder -> Int -> [Value] -> [Value]
orderValues Ordered _ vs = vs
orderValues Random  n vs =
  let key v = (v * 37) `mod` n
  in sortBy (comparing key) vs

------------------------------------------------------------
-- Sequential solver (with value order)
------------------------------------------------------------
solveSeqFirstWith :: ValueOrder -> Puzzle -> Maybe Board
solveSeqFirstWith ord p = go (givens p)
  where
    n = size p

    go :: Board -> Maybe Board
    go board =
      case chooseNext p board of
        Nothing      -> Just board
        Just (_, []) -> Nothing
        Just (pos,vs0) ->
          let vs = orderValues ord n vs0
          in firstJust [ go (M.insert pos v board) | v <- vs ]


-- solveSeq :: Puzzle -> [Board]
-- solveSeq = solveSeqWith Ordered

------------------------------------------------------------
-- Parallel solver: depth-based task parallelism + value order
------------------------------------------------------------

solveParFirstWith :: Int -> ValueOrder -> Puzzle -> Maybe Board
solveParFirstWith parDepth ord p = go 0 (givens p)
  where
    n = size p

    go :: Int -> Board -> Maybe Board
    go depth board =
      case chooseNext p board of
        Nothing      -> Just board
        Just (_, []) -> Nothing
        Just (pos,vs0) ->
          let vs = orderValues ord n vs0
              subcalls = [ go (depth + 1) (M.insert pos v board) | v <- vs ]
              results =
                if depth < parDepth
                  then subcalls `using` parList rseq
                  else subcalls
          in firstJust results


-- solvePar :: Int -> Puzzle -> [Board]
-- solvePar d = solveParWith d Ordered

------------------------------------------------------------
-- Pretty printing
------------------------------------------------------------
prettyBoard :: Int -> Board -> String
prettyBoard n b =
  concat [ printRow r ++ "\n" | r <- [0 .. n-1] ]
  where
    printRow r =
      concat [ cell r c ++ " " | c <- [0 .. n-1] ]

    cell r c =
      case M.lookup (r,c) b of
        Just v  -> show v
        Nothing -> "."

------------------------------------------------------------
-- Parsing
------------------------------------------------------------

-- Delete space at the start and the end
trim :: String -> String
trim s = dropWhile (== ' ') $ reverse $ dropWhile (== ' ') $ reverse s

-- only get number and comma for position
readPos :: String -> (Int,Int)
readPos s =
  let ds = filter (\c -> isDigit c || c == ',') s
      [a,b] = splitBy ',' ds
  in (read a, read b)

splitBy :: Char -> String -> [String]
splitBy _ "" = []
splitBy chr xs =
  let (a,bs) = span (/= chr) xs
  in case bs of
       []     -> [a]
       (_:rs) -> a : splitBy chr rs

-- "(r,c)=v"
parseGiven :: String -> ((Int,Int), Int)
parseGiven l =
  let (posS, valS0) = span (/='=') l
      valS = drop 1 valS0
  in (readPos posS, read valS)

-- split string at character
splitAtChar :: Char -> String -> (String,String)
splitAtChar ch s =
  let (l,_:r) = span (/= ch) s
  in (l,r)

-- "(r1,c1)<(r2,c2)" or "(r1,c1)>(r2,c2)"
parseRel :: String -> Rel
parseRel l
  | '<' `elem` l =
      let (a,b) = splitAtChar '<' l
      in Lt (readPos a) (readPos b)
  | '>' `elem` l =
      let (a,b) = splitAtChar '>' l
      in Gt (readPos a) (readPos b)
  | otherwise = error ("Bad constraint: " ++ l)

-- "size N"
parsePuzzle :: String -> IO Puzzle
parsePuzzle content = do
  let ls0 = map trim (lines content)
      ls  = filter (\x -> not (null x)) ls0
  case ls of
    [] -> die "Empty puzzle file"
    (first:rest) ->
      case words first of
        ["size", nStr] ->
          case reads nStr of
            [(n, "")] -> parseSections n rest
            _         -> die "Invalid size number"
        _ -> die "First non-empty line must be: size <N>"

-- "initial:" and "constraints:"
parseSections :: Int -> [String] -> IO Puzzle
parseSections n ls =
  case break (== "initial:") ls of
    (_, []) ->
      die "Missing initial: section"
    (_beforeInit, _initTag : rest1) ->
      let (givenLines, rest2) = break (== "constraints:") rest1
      in case rest2 of
           [] ->
             die "Missing constraints: section"
           (_conTag : relLines) ->
             let g = M.fromList (map parseGiven givenLines)
                 r = map parseRel relLines
             in pure (Puzzle n g r)

------------------------------------------------------------
-- Command-line parsing: seq/par + depth + ordered/random
------------------------------------------------------------

-- ValueOrder
parseOrder :: String -> Maybe ValueOrder
parseOrder s =
  case s of
    "ordered" -> Just Ordered
    "random"  -> Just Random
    _         -> Nothing

--   futoshiki <file>
--   futoshiki <file> seq [ordered|random]
--   futoshiki <file> par <depth> [ordered|random]
parseArgs :: IO (FilePath, RunMode)
parseArgs = do
  args <- getArgs
  prog <- getProgName
  case args of
    -- default: only filename → par depth=2 + ordered
    [file] ->
      pure (file, RunPar 2 Ordered)

    -- seq，default ordered
    [file, "seq"] ->
      pure (file, RunSeq Ordered)

    -- seq + ordered/random
    [file, "seq", ordStr] ->
      case parseOrder ordStr of
        Just o  -> pure (file, RunSeq o)
        Nothing -> die "Value order must be 'ordered' or 'random'."

    -- par depth, default ordered
    [file, "par", depthStr] ->
      case readMaybe depthStr of
        Just d | d >= 0 -> pure (file, RunPar d Ordered)
        _               -> die "Depth must be a non-negative integer."

    -- par depth + ordered/random
    [file, "par", depthStr, ordStr] ->
      case (readMaybe depthStr, parseOrder ordStr) of
        (Just d, Just o) | d >= 0 -> pure (file, RunPar d o)
        (Nothing, _)   -> die "Depth must be a non-negative integer."
        (_, Nothing)   -> die "Value order must be 'ordered' or 'random'."

    _ -> do
      putStrLn "Usage:"
      putStrLn $ "  " ++ prog ++ " <puzzlefile>"
      putStrLn $ "  " ++ prog ++ " <puzzlefile> seq [ordered|random]"
      putStrLn $ "  " ++ prog ++ " <puzzlefile> par <depth> [ordered|random]"
      die ""

------------------------------------------------------------
-- MAIN
------------------------------------------------------------

main :: IO ()
main = do
  (file, mode) <- parseArgs             

  txt    <- readFile file               
  puzzle <- parsePuzzle txt             -- parsing

  putStrLn $ "Puzzle file: " ++ file    
  putStrLn $ "Mode: " ++ show mode

  -- Start the runtime timer
  start <- getCurrentTime

  -- Get solution (thunks)
  let mSol = case mode of
        RunSeq ord     -> solveSeqFirstWith ord puzzle
        RunPar dep ord -> solveParFirstWith dep ord puzzle

  let forced = case mSol of
        Nothing -> "No solution."
        Just b  -> prettyBoard (size puzzle) b

  length forced `seq` return ()

  -- Stop timer
  end <- getCurrentTime

  let dt = realToFrac (diffUTCTime end start) :: Double

  -- Print the result
  putStrLn $ "Runtime (first solution): " ++ show dt ++ " seconds"
  case mSol of
    Nothing -> putStrLn "No solution found."
    Just b  -> do
      putStrLn "Found one solution:"
      putStrLn (prettyBoard (size puzzle) b)


  -- Print each solution
--   putStrLn (unlines (map (prettyBoard (size puzzle)) sols))
