{-# OPTIONS_GHC -Wall #-}

{-
Name:
Uni:

Parallel Futoshiki Solver (task parallelism on upper levels)
-}

import qualified Data.Map.Strict as M
import           Data.Map.Strict (Map)
import           Data.List       (minimumBy)
import           Data.Ord        (comparing)
import           System.Environment (getArgs, getProgName)
import           System.Exit (die)
import           Data.Char (isDigit)

import           Control.Parallel.Strategies (using, parList, rdeepseq)
import           Control.DeepSeq (NFData(..))

------------------------------------------------------------
-- Types
------------------------------------------------------------

type Pos   = (Int, Int)
type Value = Int

data Rel
  = Lt Pos Pos
  | Gt Pos Pos
  deriving (Show, Eq)

type Board = Map Pos Value

data Puzzle = Puzzle
  { size      :: Int
  , givens    :: Board
  , relations :: [Rel]
  } deriving (Show)

------------------------------------------------------------
-- NFData instances for parallel evaluation
------------------------------------------------------------

instance NFData Rel where
  rnf (Lt p1 p2) = rnf p1 `seq` rnf p2
  rnf (Gt p1 p2) = rnf p1 `seq` rnf p2
-- Map 已经在 containers 里有 NFData 实例了，这里不要再声明

------------------------------------------------------------
-- Row/col helpers
------------------------------------------------------------

rowValues :: Int -> Board -> Int -> [Value]
rowValues n board r =
  [ v
  | c <- [0 .. n-1]
  , Just v <- [M.lookup (r,c) board]
  ]

colValues :: Int -> Board -> Int -> [Value]
colValues n board c =
  [ v
  | r <- [0 .. n-1]
  , Just v <- [M.lookup (r,c) board]
  ]

------------------------------------------------------------
-- Relation checking
------------------------------------------------------------

okOneRel :: Board -> Rel -> Bool
okOneRel b (Lt p1 p2) =
  case (M.lookup p1 b, M.lookup p2 b) of
    (Just v1, Just v2) -> v1 < v2
    _                  -> True

okOneRel b (Gt p1 p2) =
  case (M.lookup p1 b, M.lookup p2 b) of
    (Just v1, Just v2) -> v1 > v2
    _                  -> True

respectsRelations :: [Rel] -> Board -> Pos -> Value -> Bool
respectsRelations rels board pos v =
  let b' = M.insert pos v board
  in all (okOneRel b') rels

------------------------------------------------------------
-- MRV heuristic
------------------------------------------------------------

candidatesFor :: Puzzle -> Board -> Pos -> [Value]
candidatesFor p board (r,c) =
  [ v
  | v <- [1 .. size p]
  , v `notElem` rowValues (size p) board r
  , v `notElem` colValues (size p) board c
  , respectsRelations (relations p) board (r,c) v
  ]

unfilledPositions :: Puzzle -> Board -> [Pos]
unfilledPositions p board =
  [ (r,c)
  | r <- [0 .. size p - 1]
  , c <- [0 .. size p - 1]
  , M.notMember (r,c) board
  ]

chooseNext :: Puzzle -> Board -> Maybe (Pos, [Value])
chooseNext p board =
  case unfilledPositions p board of
    [] -> Nothing
    ps ->
      let cs = [ (pos, candidatesFor p board pos)
               | pos <- ps
               ]
      in Just (minimumBy (comparing (length . snd)) cs)

------------------------------------------------------------
-- Sequential solver (for comparison)
------------------------------------------------------------

solveSeq :: Puzzle -> [Board]
solveSeq p = go (givens p)
  where
    go :: Board -> [Board]
    go board =
      case chooseNext p board of
        Nothing       -> [board]
        Just (_, [])  -> []
        Just (pos,vs) ->
          concat [ go (M.insert pos v board) | v <- vs ]

------------------------------------------------------------
-- Parallel solver: parDepth = how many levels to parallelize
------------------------------------------------------------

solvePar :: Int -> Puzzle -> [Board]
solvePar parDepth p = go 0 (givens p)
  where
    go :: Int -> Board -> [Board]
    go depth board =
      case chooseNext p board of
        Nothing       -> [board]
        Just (_, [])  -> []
        Just (pos,vs) ->
          let subcalls =
                [ go (depth + 1) (M.insert pos v board)
                | v <- vs
                ]
          in if depth < parDepth
               then concat (subcalls `using` parList rdeepseq)
               else concat subcalls

------------------------------------------------------------
-- Pretty print
------------------------------------------------------------

prettyBoard :: Int -> Board -> String
prettyBoard n b =
  unlines
    [ unwords
        [ maybe "." show (M.lookup (r,c) b)
        | c <- [0 .. n-1]
        ]
    | r <- [0 .. n-1]
    ]

------------------------------------------------------------
-- Parsing（和 basic 相同）
------------------------------------------------------------

trim :: String -> String
trim = dropWhile (== ' ') . reverse . dropWhile (== ' ') . reverse

readPos :: String -> (Int,Int)
readPos s =
  let ds = filter (\c -> isDigit c || c == ',') s
      [a,b] = splitBy ',' ds
  in (read a, read b)

splitBy :: Char -> String -> [String]
splitBy _ "" = []
splitBy ch xs =
  let (a,bs) = span (/= ch) xs
  in case bs of
       []     -> [a]
       (_:rs) -> a : splitBy ch rs

parseGiven :: String -> ((Int,Int), Int)
parseGiven l =
  let (posS, valS0) = span (/='=') l
      valS = drop 1 valS0
  in (readPos posS, read valS)

splitAtChar :: Char -> String -> (String,String)
splitAtChar ch s =
  let (l,_:r) = span (/= ch) s
  in (l,r)

parseRel :: String -> Rel
parseRel l
  | '<' `elem` l =
      let (a,b) = splitAtChar '<' l
      in Lt (readPos a) (readPos b)
  | '>' `elem` l =
      let (a,b) = splitAtChar '>' l
      in Gt (readPos a) (readPos b)
  | otherwise = error ("Bad constraint: " ++ l)

parsePuzzle :: String -> IO Puzzle
parsePuzzle content = do
  let ls0 = map trim (lines content)
      ls  = filter (not . null) ls0
  case ls of
    [] -> die "Empty puzzle file"
    (first:rest) ->
      case words first of
        ["size", nStr] ->
          case reads nStr of
            [(n, "")] -> parseSections n rest
            _         -> die "Invalid size number"
        _ -> die "First non-empty line must be: size <N>"

parseSections :: Int -> [String] -> IO Puzzle
parseSections n ls =
  case break (== "initial:") ls of
    (_, []) ->
      die "Missing initial: section"
    (_beforeInit, _initTag : rest1) ->
      let (givenLines, rest2) = break (== "constraints:") rest1
      in case rest2 of
           [] ->
             die "Missing constraints: section"
           (_conTag : relLines) ->
             let g = M.fromList (map parseGiven givenLines)
                 r = map parseRel relLines
             in pure (Puzzle n g r)

------------------------------------------------------------
-- MAIN （支持 seq / par）
------------------------------------------------------------

main :: IO ()
main = do
  args <- getArgs
  prog <- getProgName
  case args of
    -- 默认并行，深度 2
    [file] -> do
      txt    <- readFile file
      puzzle <- parsePuzzle txt
      let sols = solvePar 2 puzzle
      putStrLn "Mode: parallel (depth = 2)"
      putStrLn ("Number of solutions: " ++ show (length sols))
      putStrLn (unlines (map (prettyBoard (size puzzle)) sols))

    [file, "seq"] -> do
      txt    <- readFile file
      puzzle <- parsePuzzle txt
      let sols = solveSeq puzzle
      putStrLn "Mode: sequential"
      putStrLn ("Number of solutions: " ++ show (length sols))
      putStrLn (unlines (map (prettyBoard (size puzzle)) sols))

    [file, "par", depthStr] ->
      case reads depthStr of
        [(d, "")] | d >= 0 -> do
          txt    <- readFile file
          puzzle <- parsePuzzle txt
          let sols = solvePar d puzzle
          putStrLn ("Mode: parallel (depth = " ++ show d ++ ")")
          putStrLn ("Number of solutions: " ++ show (length sols))
          putStrLn (unlines (map (prettyBoard (size puzzle)) sols))
        _ -> die "Depth must be a non-negative integer"

    _ -> do
      putStrLn ("Usage:")
      putStrLn ("  " ++ prog ++ " <puzzlefile>")
      putStrLn ("  " ++ prog ++ " <puzzlefile> seq")
      putStrLn ("  " ++ prog ++ " <puzzlefile> par <depth>")
      die ""
