# Futoshiki Solver (Seq + Par) — Reproducible Experiments

This repository contains a Futoshiki puzzle solver implemented in Haskell, including:

* **Sequential solver** (baseline)
* **Parallel solver** (configurable parallel depth)
* Two value-ordering (selection) strategies:

  * `ordered`
  * `random`

It also includes scripts and logged outputs to reproduce all experiments used in the report.

---

## 1. Repository Structure

Key directories/files:

* `futoshiki.hs` / `futoshiki` : main solver (executable)
* `puzzles/` : puzzle inputs

  * `puzzles/curated/` : human-designed puzzles and density variants (e.g., `7_0.txt`, `7_75.txt`)
  * `puzzles/random_gen/` : randomly generated puzzles matched by size
* `scripts/` : experiment runner scripts (baseline, depth sweep, scaling, etc.)
* `results/`

  * `results/raw/` : raw per-run outputs (one file per run)
  * `results/csv/` : aggregated CSV tables used for plots

---

## 2. Requirements

### Toolchain

* **GHC + Stack**
* Recommended: a multi-core machine for scaling experiments

Check Stack:

```bash
stack --version
```

---

## 3. Build

From repo root:

```bash
stack build
```

(Optional) clean build:

```bash
stack clean
stack build
```

---

## 4. Run the Solver (Manual)

### General CLI Pattern

The solver is invoked via Stack like:

```bash
stack exec futoshiki -- <PUZZLE_FILE> <MODE> <ARG...> +RTS -N<CORES> -RTS
```

Where:

* `<PUZZLE_FILE>`: e.g. `puzzles/curated/7_easy.txt`
* `<MODE>`:

  * `seq` for sequential
  * `par` for parallel
* `<ARG...>` depends on mode:

  * `seq <ordered|random>`
  * `par <parDepth> <ordered|random>`

### Examples

#### Sequential + ordered

```bash
stack exec futoshiki -- puzzles/curated/7_easy.txt seq ordered +RTS -N1 -RTS
```

#### Sequential + random

```bash
stack exec futoshiki -- puzzles/curated/7_easy.txt seq random +RTS -N1 -RTS
```

#### Parallel (parDepth = 3) + ordered, using 4 cores

```bash
stack exec futoshiki -- puzzles/curated/7_easy.txt par 3 ordered +RTS -N4 -RTS
```

#### Parallel (parDepth = 3) + random, using 8 cores

```bash
stack exec futoshiki -- puzzles/curated/7_easy.txt par 3 random +RTS -N8 -RTS
```

### Output

A typical run prints:

* puzzle file + mode
* runtime (time to first solution)
* one solution (since we stop after the first solution)

---

## 5. Reproduce All Experiments (Recommended)

All experiments are runnable via scripts in `scripts/`.

Make them executable (one-time):

```bash
chmod +x scripts/*.sh
```

### Experiment 1 — Baseline (Seq: ordered vs random)

Runs sequential solver on the baseline puzzle and repeats multiple times.

```bash
./scripts/run_baseline.sh
```

Outputs:

* Raw logs: `results/raw/baseline/`
* Aggregated table: `results/csv/baseline.csv`

---

### Experiment 2 — Parallel Depth Sweep (parDepth)

Sweeps `parDepth` values for the parallel solver (fixed cores).

```bash
./scripts/run_depth.sh
```

Outputs:

* Raw logs: `results/raw/depth_sweep/`
* Aggregated table: `results/csv/depth_sweep.csv`

---

### Experiment 3 — Scaling with Number of Cores

Runs parallel solver with fixed `parDepth` while varying `+RTS -N`.

```bash
./scripts/run_scaling.sh
```

Outputs:

* Raw logs: `results/raw/scaling/`
* Aggregated table: `results/csv/scaling.csv`

---

### Experiment 4 — Parallel Value Ordering (ordered vs random)

Compares ordered vs random under the **parallel** solver (same depth and cores).

```bash
./scripts/run_par_order_compare.sh
```

Outputs:

* Raw logs: `results/raw/par_order_compare/`
* Aggregated table: `results/csv/par_order_compare.csv`

---

## 5. Puzzle Inputs

All puzzles are plain-text inputs in:

* `puzzles/curated/` (human-designed + density variants)
* `puzzles/random_gen/` (random puzzles)

Examples:

* `puzzles/curated/7_easy.txt` : benchmark puzzle used across depth + scaling experiments
* `puzzles/curated/7_0.txt ... 7_75.txt` : constraint density variants (constructed from a known solution)
* `puzzles/curated/6x6.txt, 7x7.txt, 9x9.txt` : curated boards for human vs random comparisons
* `puzzles/random_gen/6x6.txt, 7x7.txt, 9x9.txt` : random counterparts for human vs random comparisons

---

## 6. Where Results Go

* **Raw run outputs** (one file per run):

  * `results/raw/baseline/`
  * `results/raw/depth_sweep/`
  * `results/raw/scaling/`
  * `results/raw/par_order_compare/`

* **CSV summaries** (used for plots/tables):

  * `results/csv/baseline.csv`
  * `results/csv/depth_sweep.csv`
  * `results/csv/scaling.csv`
  * `results/csv/par_order_compare.csv`

---

## 7. (Optional) Eventlog / ThreadScope Profiling

If you want to generate an eventlog (for ThreadScope visualization), run with `-ls` and keep `+RTS`.

Example:

```bash
stack exec futoshiki -- puzzles/curated/7_easy.txt par 3 ordered +RTS -N4 -ls -RTS
```

This produces an `.eventlog` file (e.g., `futoshiki.eventlog`) that can be opened in ThreadScope.

---

## 8. Reproducibility Notes

* All runtimes reported are **time to first solution** (the solver stops after finding the first valid solution).
* For fair comparisons, scripts fix:

  * puzzle file
  * solver mode (seq/par)
  * `parDepth` (when applicable)
  * core count via `+RTS -N`
  * number of repeated runs
* Some experiments (e.g., `random` ordering) may have variance; use multiple runs and CSV aggregation.

---

## 9. Quick Start

If I only want to verify everything works:

```bash
stack build
stack exec futoshiki -- puzzles/curated/7_easy.txt seq ordered +RTS -N1 -RTS
./scripts/run_baseline.sh
```