module SequentialMCTS (rootNode, mctsAlgo, bestMove) where

import Board
import System.Random (randomRIO)
import Data.List (maximumBy)

data Node = Node {
    nodeId    :: Int,
    board     :: Board,
    color     :: Color,
    children  :: [Node],
    numVisits :: Int,
    numWins   :: Int 
} deriving (Show, Eq)

rootNode :: Board -> Color -> Node
rootNode b c = Node {
    nodeId    = 0,
    board     = b,
    color     = c,
    children  = [],
    numVisits = 0,
    numWins   = 0 }

otherColor :: Color -> Color
otherColor Red    = Yellow
otherColor Yellow = Red

expand :: Node -> Int -> ([Node], Int)
expand parent startId =
    let moves = availableCols (board parent)
        (cs, finalId) = foldl makeNode ([], startId) moves
    in (reverse cs, finalId)
    where
        makeNode (acc, currentId) move = 
            case placeTile (board parent) move (color parent) of
                Just newBoard -> 
                    let newNode = Node { 
                        nodeId    = currentId,
                        board     = newBoard,
                        color     = otherColor (color parent),
                        children  = [],
                        numVisits = 0,
                        numWins   = 0
                        }
                    in (newNode : acc, currentId + 1)
                Nothing -> (acc, currentId)  -- Skip invalid moves, don't increment ID

play :: Board -> Color -> IO Color
play b c = do
    case checkWin b of
        Just winner -> return winner
        Nothing ->
            if length (availableCols b) == 0
            then return c 
            else do
                let choices = availableCols b
                choice <- randomRIO (0, length choices - 1)
                let move = choices !! choice
                case placeTile b move c of
                    Just newBoard -> play newBoard (otherColor c)
                    Nothing -> play b c

mctsAlgo :: Node -> Int -> Int -> IO Node
mctsAlgo node _ 0 = do
    return node
mctsAlgo node nextId n = do
    (newNode, newNextId) <- oneMcts node nextId
    mctsAlgo newNode newNextId (n - 1)

score :: Int -> Node -> Double
score parentVisits node
    | numVisits node == 0 = (1 / 0) -- Set equal to infinity. Exlore unvisited!
    | otherwise = exploitation + exploration
    where
        exploitation = fromIntegral (numWins node) / fromIntegral (numVisits node)
        -- Chosing sqrt 2 as exploration parameter
        exploration = sqrt (2 * log (fromIntegral parentVisits) 
                                / fromIntegral (numVisits node))
    
selectNextNode :: Node -> Node
selectNextNode node = 
    case children node of
        [] -> node
        childList -> maximumBy compareScore childList
    where
        parentVisits = max 1 (numVisits node) -- prevents log(0) if numVisits was 0
        compareScore node1 node2 = compare (score parentVisits node1) (score parentVisits node2)

selectPath :: Node -> [Node]
selectPath node
    | null (children node) = [node]
    | otherwise = node : selectPath (selectNextNode node)

updateNode :: Color -> Node -> Node
updateNode winner node = node {
    numVisits = numVisits node + 1,
    numWins = numWins node + (if winner == (color node) then 1 else 0)
}

oneMcts :: Node -> Int -> IO (Node, Int)
oneMcts root nextId = do
    let path = selectPath root
        leaf = last path
    let (newChildren, finalId) = expand leaf nextId
    if null newChildren
    then do
        winner <- play (board leaf) (color leaf)
        let result = backpropagate winner path
        return (result, finalId)
    else do
        let childToSimulate = head newChildren
        winner <- play (board childToSimulate) (color childToSimulate)
        let updatedChild = updateNode winner childToSimulate
            updatedChildren = updatedChild : tail newChildren
            updatedLeaf = updateNode winner (leaf { children = updatedChildren })
        let fullPath = init path ++ [updatedLeaf]
        let result = backpropagate winner fullPath
        return (result, finalId)

backpropagate :: Color -> [Node] -> Node
backpropagate winner path =
    foldr updateNodeAlongPath (last path) (init path)
  where
    updateNodeAlongPath parent childWithUpdated =
        let updatedParent = parent {
                numVisits = numVisits parent + 1,
                numWins   = numWins parent + if winner == color parent then 1 else 0,
                children  = map (\c -> if nodeId c == nodeId childWithUpdated then childWithUpdated else c) (children parent)
            }
        in updatedParent

bestMove :: Node -> Maybe Node
bestMove root =
    case children root of
        [] -> Nothing
        childList -> Just $ maximumBy compareVisits childList
    where
        compareVisits node1 node2 = compare (numVisits node1) (numVisits node2)