module Board (Color(..), Board(..), createEmptyBoard, availableCols, placeTile, checkWin) where

import Data.List (transpose)

data Color = Red | Yellow
    deriving (Eq, Show)

data Spot = Empty | Taken Color
    deriving (Eq, Show)

{- Note for clarity: the 7 column x 6 row board will be represented with a 2D
list where there are 7 elements in the outer list representing the 7 columns
from left to right. Each element is a 6 element list representing the rows
from bottom to top. (0,0) = bottom left, (0,5) = top left, (6,0) = bottom
right, (6,5) = top right. -}
newtype Board = Board [[Spot]]
    deriving (Eq, Show)

createEmptyBoard :: Board
createEmptyBoard = Board (replicate 7 (replicate 6 Empty))

availableCols :: Board -> [Int]
availableCols (Board board) = 
    [i | i <- [0 .. length board - 1], elem Empty (board !! i)]

{- Attempt to put a tile in a column. If it is an illegal move, it will return
Nothing. Otherwise, it will return the new column. -}
putInCol :: Color -> [Spot] -> Maybe [Spot]
putInCol _ []                 = Nothing
putInCol color (Empty : rest) = Just (Taken color : rest)
putInCol color (spot : rest)  = fmap (spot :) (putInCol color rest)

{- Attempt to place a tile in the board. If it is out of the column range or
an illegal move (determined by putInCol), it will return Nothing. Otherwise,
it will return the new board. -}
placeTile :: Board -> Int -> Color -> Maybe Board
placeTile (Board board) col color 
    | col < 0 || col >= length board = Nothing
    | otherwise = case splitAt col board of
        (left, column : right) ->
            case putInCol color column of
                Nothing -> Nothing
                Just newCol -> Just (Board (left ++ newCol : right))
        _ -> Nothing

check4 :: [Spot] -> Maybe Color
check4 [] = Nothing
check4 (s1 : s2 : s3 : s4 : rest) =
    case (s1, s2, s3, s4) of 
        (Taken c1, Taken c2, Taken c3, Taken c4)
            | (c1 == c2 && c2 == c3 && c3 == c4) -> Just c1
        _ -> check4 (s2 : s3 : s4 : rest)
check4 _ = Nothing

checkHorizontal :: Board -> Maybe Color
checkHorizontal (Board board) = checkRow (transpose board)
    where
        checkRow []           = Nothing
        checkRow (row : rows) = case check4 row of
                                    Just color -> Just color
                                    Nothing -> checkRow rows

checkVertical :: Board -> Maybe Color
checkVertical (Board board) = checkCol board
    where
        checkCol []           = Nothing
        checkCol (col : cols) = case check4 col of
                                    Just color -> Just color
                                    Nothing -> checkCol cols

getDiags :: Board -> [[Spot]]
getDiags (Board board) =
    let diagsLToR = do
            col <- [0..3]
            row <- [0..2]
            return [board !! (col + i) !! (row + i) | i <- [0..3]]
        diagsRToL = do
            col <- [0..3]
            row <- [3..5]
            return [board !! (col + i) !! (row - i) | i <- [0..3]]
    in diagsLToR ++ diagsRToL

checkDiagonal :: Board -> Maybe Color
checkDiagonal board = checkDiag (getDiags board)
    where
        checkDiag []             = Nothing
        checkDiag (diag : diags) = case check4 diag of
                                      Just color -> Just color
                                      Nothing -> checkDiag diags

checkWin :: Board -> Maybe Color
checkWin board = 
    case checkHorizontal board of
        Just color -> Just color
        Nothing ->
            case checkVertical board of
                Just color -> Just color
                Nothing -> checkDiagonal board