module MaxN
    ( maxN
    , maxNStrat
    , maxNStratParBudget
    ) where

import Data.List (maximumBy)
import Data.Ord  (comparing)
import Control.Parallel.Strategies (Strategy, using, parList)

import qualified Blokus as Bl

------------------------------------------------------------
-- Sequential MaxN
------------------------------------------------------------

-- | Pure sequential MaxN (no parallelism).
maxN :: Int -> Bl.GameState -> Bl.Scores
maxN depth gs
    | depth == 0 || Bl.isTerminal gs = Bl.evaluate gs
    | otherwise =
        case Bl.validMoves gs of
            [] ->
                let gsPass = Bl.passTurn gs
                in maxN depth gsPass
            moves ->
                let p      = Bl.gsPlayerToMove gs
                    scores = [ maxN (depth - 1) (Bl.applyValidMove gs m)
                             | m <- moves
                             ]
                in maximumBy (comparing (Bl.scoreOfPlayer p)) scores

------------------------------------------------------------
-- Parallel MaxN with a Strategy
------------------------------------------------------------

-- | Parallel MaxN parameterized by a 'Strategy' for the list of child scores.
--
-- Example strategies:
--   * @parList rseq@
--   * @parList rpar@
--   * @parList rdeepseq@
maxNStrat
    :: Strategy [Bl.Scores]  -- ^ Strategy applied to the list of child scores.
    -> Int                   -- ^ Search depth.
    -> Bl.GameState
    -> Bl.Scores
maxNStrat strat depth gs
    | depth == 0 || Bl.isTerminal gs = Bl.evaluate gs
    | otherwise =
        case Bl.validMoves gs of
            [] ->
                let gsPass = Bl.passTurn gs
                in maxNStrat strat depth gsPass
            moves ->
                let p      = Bl.gsPlayerToMove gs
                    scores = [ maxNStrat strat (depth - 1) (Bl.applyValidMove gs m)
                             | m <- moves
                             ] `using` strat
                in maximumBy (comparing (Bl.scoreOfPlayer p)) scores

------------------------------------------------------------
-- Parallel MaxN with Strategy + parBudget
------------------------------------------------------------

-- | Parallel MaxN with a 'Strategy' and a parallelization budget.
--
-- 'parBudget' controls how many levels from the current node down will
-- use the strategy; below that the search is sequential.
--
-- Typical usage:
--   * @maxNStratParBudget (parList rseq)     depth 1 gs@
--   * @maxNStratParBudget (parList rdeepseq) depth 2 gs@
maxNStratParBudget
    :: Strategy [Bl.Scores]  -- ^ Strategy for the list of child scores.
    -> Int                   -- ^ Search depth.
    -> Int                   -- ^ Parallelization budget (levels).
    -> Bl.GameState
    -> Bl.Scores
maxNStratParBudget strat depth parBudget gs
    | depth == 0 || Bl.isTerminal gs = Bl.evaluate gs
    | otherwise =
        case Bl.validMoves gs of
            [] ->
                let gsPass = Bl.passTurn gs
                in maxNStratParBudget strat depth parBudget gsPass
            moves ->
                let p         = Bl.gsPlayerToMove gs
                    evalChild m =
                        maxNStratParBudget strat (depth - 1) (parBudget - 1)
                                           (Bl.applyValidMove gs m)
                    scoresBase = [ evalChild m | m <- moves ]
                    scores     =
                        if parBudget > 0
                           then scoresBase `using` strat
                           else scoresBase
                in maximumBy (comparing (Bl.scoreOfPlayer p)) scores
