#!/usr/bin/env bash
set -euo pipefail

usage() {
  cat <<EOF
Usage: $0 [MODE] [EVENTS]

MODE:
  all        Run seq, par, and parBudget (default)
  par        Run only par (no parBudget)
  parBudget  Run only parBudget

EVENTS:
  events     Generate ThreadScope eventlogs (default)
  noevents   Do not generate eventlogs

Examples:
  $0                  # run all modes, with eventlogs
  $0 par noevents     # run only par, no eventlogs
  $0 parBudget events # run only parBudget, with eventlogs
EOF
}

if [[ "${1:-}" == "-h" || "${1:-}" == "--help" ]]; then
  usage
  exit 0
fi

MODE="${1:-all}"       # all | par | parBudget
EVENTS="${2:-events}"  # events | noevents

STACK_RUN="stack run --"
LOGDIR="logs"
mkdir -p "$LOGDIR"

DEPTHS=(5)
CORES=(10)
STRATS=(rpar rseq rdeepseq)
PAR_BUDGETS=(2 3)

# Build once
stack build

run_case() {
  local impl="$1" depth="$2" cores="$3" strat="$4" budget="$5"

  local tag="d${depth}"
  [[ "$strat" != "-" ]] && tag="${tag}-${strat}"
  [[ "$budget" != "-" ]] && tag="${tag}-b${budget}"
  tag="${tag}-n${cores}"

  local stats_log="${LOGDIR}/${impl}-${tag}-stats.log"
  local time_log="${LOGDIR}/${impl}-${tag}-time.log"
  local event_log="${LOGDIR}/${impl}-${tag}.eventlog"

  echo "=== ${impl} ${tag} ==="

  rm -f blokus-exe.eventlog

  # Build argument list for Main
  if [[ "$impl" == "seq" ]]; then
    app_args=("seq" "$depth")
  elif [[ "$impl" == "par" ]]; then
    app_args=("par" "$depth" "$strat")
  elif [[ "$impl" == "parBudget" ]]; then
    app_args=("parBudget" "$depth" "$strat" "$budget")
  else
    echo "Unknown impl: $impl" >&2
    exit 1
  fi

  # RTS options: include -l only if EVENTS == events
  local rts_opts=("-N${cores}" "-s")
  if [[ "$EVENTS" == "events" ]]; then
    rts_opts+=("-l")
  fi

  # Use shell built-in time; stdout -> time_log, RTS stats -> stats_log
  { time $STACK_RUN "${app_args[@]}" +RTS "${rts_opts[@]}" -RTS \
        1> "$time_log" \
        2> "$stats_log"; } 2>> "$time_log"

  if [[ "$EVENTS" == "events" && -f blokus-exe.eventlog ]]; then
    mv blokus-exe.eventlog "$event_log"
  fi
}

# Sequential baseline (only when MODE == all)
if [[ "$MODE" == "all" ]]; then
  for depth in "${DEPTHS[@]}"; do
    run_case "sequential" "$depth" 1 "-" "-"
  done
fi

# par (no parBudget): all strategies and core counts
if [[ "$MODE" == "all" || "$MODE" == "par" ]]; then
  for depth in "${DEPTHS[@]}"; do
    for cores in "${CORES[@]}"; do
      for strat in "${STRATS[@]}"; do
        run_case "par" "$depth" "$cores" "$strat" "-"
      done
    done
  done
fi

# parBudget: all strategies, budgets 1 and 2, all core counts
if [[ "$MODE" == "all" || "$MODE" == "parBudget" ]]; then
  for depth in "${DEPTHS[@]}"; do
    for cores in "${CORES[@]}"; do
      for strat in "${STRATS[@]}"; do
        for budget in "${PAR_BUDGETS[@]}"; do
          run_case "parBudget" "$depth" "$cores" "$strat" "$budget"
        done
      done
    done
  done
fi
