module Main (main) where

import System.Environment (getArgs, getProgName)
import System.Exit        (die)

import Control.Parallel.Strategies (Strategy, parList, rseq, rpar, rdeepseq)

import qualified Blokus as Bl
import qualified MaxN   as Mx

main :: IO ()
main = do
    args <- getArgs
    case args of
        ["seq", depthStr] -> runSeq (read depthStr)
        ["par", depthStr, stratStr] -> runPar (read depthStr) (parseStrat stratStr)
        ["parBudget", depthStr, stratStr, budgetStr] -> runParBudget (read depthStr) (parseStrat stratStr) (read budgetStr)
        _ -> usage

  where
    -- Strategy parser: "rseq" or "rdeepseq"
    parseStrat :: String -> Strategy [Bl.Scores]
    parseStrat "rseq"     = parList rseq
    parseStrat "rpar"     = parList rpar
    parseStrat "rdeepseq" = parList rdeepseq
    parseStrat other      = error ("Unknown strategy: " ++ other ++ " (use rpar, rseq or rdeepseq)")

    runSeq :: Int -> IO ()
    runSeq depth = do
        let gs0  = Bl.initGameState
            best = Mx.maxN depth gs0
        putStrLn "Impl: seq"
        putStrLn $ "Depth: " ++ show depth
        putStrLn $ "Evaluation: " ++ show best

    runPar :: Int -> Strategy [Bl.Scores] -> IO ()
    runPar depth strat = do
        let gs0  = Bl.initGameState
            best = Mx.maxNStrat strat depth gs0
        putStrLn "Impl: par"
        putStrLn $ "Depth: " ++ show depth
        putStrLn $ "Strategy: parList <strat>"
        putStrLn $ "Evaluation: " ++ show best

    runParBudget :: Int -> Strategy [Bl.Scores] -> Int -> IO ()
    runParBudget depth strat budget = do
        let gs0  = Bl.initGameState
            best = Mx.maxNStratParBudget strat depth budget gs0
        putStrLn "Impl: parBudget"
        putStrLn $ "Depth: " ++ show depth
        putStrLn $ "Strategy: parList <strat>"
        putStrLn $ "ParBudget: " ++ show budget
        putStrLn $ "Evaluation: " ++ show best

    usage :: IO a
    usage = do
        progName <- getProgName
        die $ unlines
          [ "Usage:"
          , "  " ++ progName ++ " seq        <depth>"
          , "  " ++ progName ++ " par        <depth> <rpar|rseq|rdeepseq>"
          , "  " ++ progName ++ " parBudget  <depth> <rpar|rseq|rdeepseq> <parBudget>"
          ]
