/*
 * Copyright (c) 2000, Ping Pan and Henning Schulzrinne
 *	All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Initially written by Ping Pan, Columbia University/Bell Labs, February 2000.
 * We acknowledge Bell Labs for support. Please forward bug fixes,enhancements 
 * and questions to pingpan@cs.columbia.edu.
 */

#ifndef _IPOPT_VAR_H_
#define _IPOPT_VAR_H_

/* used as an input for ipopt_input() */
#define IPOPT_LOCAL	1

/*
 * IP Option Control Block:
 */
struct ipoptpcb {
	LIST_ENTRY(ipoptpcb) list;
	struct	socket *ocb_socket;	/* pointer to socket */
	struct	sockproto ocb_proto;	/* protocol family, protocol */
	int	ocb_flags;		/* socket options, etc. */
};

#define sotoipoptcb(so)		((struct ipoptpcb *)(so)->so_pcb)


/* space alloacted to a raw socket for IP option processing */
#define IPOPTSNDQ	8192
#define IPOPTRCVQ	8192


/* flags in ocb_flags: */
#define OCB_RECVIF	0x0001		/* record incoming if */
#define OCB_RECVRA	0x0002		/* recv all RA */
#define OCB_RECVRSVP	0x0004		/* recv RSVP */
#define OCB_RECVRTCP	0x0008		/* recv RTCP */
#define OCB_RECVLOCAL	0x0010		/* recv local pkt's */

#define OCB_RECVIF_SET(x)		((x)->ocb_flags & OCB_RECVIF)
#define OCB_RA_SET(x)			((x)->ocb_flags & OCB_RECVRA)
#define OCB_RSVP_SET(x)			((x)->ocb_flags & OCB_RECVRSVP)
#define OCB_RTCP_SET(x)			((x)->ocb_flags & OCB_RECVRTCP)
#define OCB_RECVLOCAL_SET(x)		((x)->ocb_flags & OCB_RECVLOCAL)

#define OCB_SET_RECVIF(x)		{(x)->ocb_flags |= OCB_RECVIF;}
#define OCB_SET_RA(x)			{(x)->ocb_flags |= OCB_RECVRA;}
#define OCB_SET_RSVP(x)			{(x)->ocb_flags |= OCB_RECVRSVP;}
#define OCB_SET_RTCP(x)			{(x)->ocb_flags |= OCB_RECVRTCP;}
#define OCB_SET_RECVLOCAL(x)		{(x)->ocb_flags |= OCB_RECVLOCAL;}

#define OCB_BIT(x, bit)			((x)->ocb_flags & bit ? 1 : 0)


/* With this new extension, more users may start to experiment with
 * IP options for active network, or in-line signaling protocols,
 * To make sure we don't have to worry about performance issues
 * months down the line, we here make sure each IP option type
 * has its own PCB chain. 
 */

/* check LIST_HEAD in /sys/sys/queue.h
 */
struct ipoptcb_list_head {
	struct ipoptpcb	*lh_first;	/* first element */
};


/* system-level info */
struct ipopt_stat {
	/* maintain the number of socket requests */
	u_long	rr_count;
	u_long	ts_count;
	u_long	sec_count;
	u_long	lsrr_count;
	u_long	ssrr_count;
	u_long	ra_count;

	u_long	allra_count;
	u_long	rsvp_count;
	u_long	rtcp_count;

	/* if have time, need to export the info (sysctl ?) */
	u_long	null_input;
	u_long	rr_input;
	u_long	ts_input;
	u_long	sec_input;
	u_long	lsrr_input;
	u_long	ssrr_input;
	u_long	ra_input;

	u_long	no_space;
	u_long	lost_input;
};


/*
 * Used to maintain IP option data
 */
struct ipopt_mask {
	u_long	flag;		/* see below for bit mapping */
	u_long	alert_flag;
};

/* ipopt_mask: flag field */
#define IPOPT_RR_MASK		0x0001		/* 0000 0001 */
#define IPOPT_TS_MASK		0x0002		/* 0000 0010 */
#define IPOPT_SEC_MASK		0x0004		/* 0000 0100 */
#define IPOPT_LSRR_MASK		0x0008		/* 0000 1000 */
#define IPOPT_SSRR_MASK		0x0010		/* 0001 0000 */
#define IPOPT_RA_MASK		0x0020		/* 0010 0000 */

#define IPOPT_RR_OFFSET		0
#define IPOPT_TS_OFFSET		1
#define IPOPT_SEC_OFFSET	2
#define IPOPT_LSRR_OFFSET	3
#define IPOPT_SSRR_OFFSET	4
#define IPOPT_RA_OFFSET		5

#define IPOPT_MAX_OPTIONS	6

/* ipopt_mask: alert_flag field */
#define IPOPT_ALLRA_MASK	0x0001
#define IPOPT_ARSVP_MASK	0x0002
#define IPOPT_ARTCP_MASK	0x0004

/* check the flag: zero or not */
#define IPOPT_RR_SET(x)		((x)->flag & IPOPT_RR_MASK)
#define IPOPT_TS_SET(x)		((x)->flag & IPOPT_TS_MASK)
#define IPOPT_SEC_SET(x)	((x)->flag & IPOPT_SEC_MASK)
#define IPOPT_LSRR_SET(x)	((x)->flag & IPOPT_LSRR_MASK)
#define IPOPT_SSRR_SET(x)	((x)->flag & IPOPT_SSRR_MASK)
#define IPOPT_RA_SET(x)		((x)->flag & IPOPT_RA_MASK)

#define IPOPT_ALLRA_SET(x)	((x)->alert_flag & IPOPT_ALLRA_MASK)
#define IPOPT_ARSVP_SET(x)	((x)->alert_flag & IPOPT_ARSVP_MASK)
#define IPOPT_ARTCP_SET(x)	((x)->alert_flag & IPOPT_ARTCP_MASK)


/* set the flag */
#define IPOPT_SET_RR(x)		{(x)->flag |= IPOPT_RR_MASK;}
#define IPOPT_SET_TS(x)		{(x)->flag |= IPOPT_TS_MASK;}
#define IPOPT_SET_SEC(x)	{(x)->flag |= IPOPT_SEC_MASK;}
#define IPOPT_SET_LSRR(x)	{(x)->flag |= IPOPT_LSRR_MASK;}
#define IPOPT_SET_SSRR(x)	{(x)->flag |= IPOPT_SSRR_MASK;}
#define IPOPT_SET_RA(x)		{(x)->flag |= IPOPT_RA_MASK;}

#define IPOPT_SET_ALLRA(x)	{(x)->alert_flag |= IPOPT_ALLRA_MASK;}
#define IPOPT_SET_ARSVP(x)	{(x)->alert_flag |= IPOPT_ARSVP_MASK;}
#define IPOPT_SET_ARTCP(x)	{(x)->alert_flag |= IPOPT_ARTCP_MASK;}

/* clear the flag */
#define IPOPT_RESET_RR(x)	{(x)->flag &= ~IPOPT_RR_MASK;}
#define IPOPT_RESET_TS(x)	{(x)->flag &= ~IPOPT_TS_MASK;}
#define IPOPT_RESET_SEC(x)	{(x)->flag &= ~IPOPT_SEC_MASK;}
#define IPOPT_RESET_LSRR(x)	{(x)->flag &= ~IPOPT_LSRR_MASK;}
#define IPOPT_RESET_SSRR(x)	{(x)->flag &= ~IPOPT_SSRR_MASK;}
#define IPOPT_RESET_RA(x)	{(x)->flag &= ~IPOPT_RA_MASK;}

#define IPOPT_RESET_ALLRA(x)	{(x)->alert_flag &= ~IPOPT_ALLRA_MASK;}
#define IPOPT_RESET_ARSVP(x)	{(x)->alert_flag &= ~IPOPT_ARSVP_MASK;}
#define IPOPT_RESET_ARTCP(x)	{(x)->alert_flag &= ~IPOPT_ARTCP_MASK;}


/* global parameters */
extern struct domain	ipoptdomain;
extern struct protosw	ipoptsw[];
extern struct pr_usrreqs	ipopt_usrreqs;
extern struct ipopt_stat	ipoptstat;
extern struct ipopt_mask	ipopt_cfg;
extern struct ipopt_mask	ipopt_pkt;
extern u_char ipopt_protox[IPOPT_MAX_OPTIONS];

extern struct ipoptcb_list_head ipoptcb_ralist;
extern struct ipoptcb_list_head ipoptcb_rrlist;
extern struct ipoptcb_list_head ipoptcb_tslist;
extern struct ipoptcb_list_head ipoptcb_lsrrlist;
extern struct ipoptcb_list_head ipoptcb_ssrrlist;
extern struct ipoptcb_list_head ipoptcb_seclist;

/* func prototypes */
void	ipopt_init(void);
int	ipopt_ctloutput(struct socket *, struct sockopt *);
void	ipopt_map(int, u_char *, struct ip *);
int	ipopt_input(struct mbuf *, int);

void	null_input(struct mbuf *, int);
void	ra_input(struct mbuf *, int);
void	rr_input(struct mbuf *, int);
void	lsrr_input(struct mbuf *, int);
void	ssrr_input(struct mbuf *, int);
void	ts_input(struct mbuf *, int);
void	security_input(struct mbuf *, int);

void	ipopt_reg_proto(int, int);
void	ipopt_reset_cb(struct socket *);
int	ipopt_free_cb(struct socket *);
int	ipopt_accept_proto(int, struct ipoptpcb *);

void	ipopt_proto_input(struct mbuf *, int, int, struct ipoptcb_list_head *);
void	ipopt_appenctl(struct ipoptpcb *, struct mbuf **, struct ip *, struct mbuf *, int);

#endif
