/*
 * Copyright (c) 2000, Ping Pan and Henning Schulzrinne
 *	All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Initially written by Ping Pan, Columbia University/Bell Labs, February 2000.
 * We acknowledge Bell Labs for support. Please forward bug fixes,enhancements 
 * and questions to pingpan@cs.columbia.edu.
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/sysctl.h>
#include <sys/proc.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/queue.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/domain.h>
#include <sys/protosw.h>
#include <net/raw_cb.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netipopt/ipopt_var.h>

/* IPOPTION extension variables:
 */

struct ipopt_stat ipoptstat;			/* keep track of sockets */
struct ipopt_mask ipopt_cfg;			/* user config data */
struct ipopt_mask ipopt_pkt;			/* packet data */
u_char ipopt_protox[IPOPT_MAX_OPTIONS];		/* input func pointers */

static u_long	ipopt_sendspace = IPOPTSNDQ;	/* socket buf size */
static u_long	ipopt_recvspace = IPOPTRCVQ;	/* same */

struct ipoptcb_list_head ipoptcb_ralist;		/* PCB chain */
struct ipoptcb_list_head ipoptcb_rrlist;		/* PCB chain */
struct ipoptcb_list_head ipoptcb_tslist;		/* PCB chain */
struct ipoptcb_list_head ipoptcb_lsrrlist;		/* PCB chain */
struct ipoptcb_list_head ipoptcb_ssrrlist;		/* PCB chain */
struct ipoptcb_list_head ipoptcb_seclist;		/* PCB chain */


/* called during domain init: clean up the varialbe blocks, and 
 * register the new protocols to ipopt_protox[] table.
 */
void
ipopt_init()
{
	bzero((char *)&ipoptstat, sizeof(struct ipopt_stat));
	bzero((char *)&ipopt_cfg, sizeof(struct ipopt_mask));
	bzero((char *)&ipopt_protox, IPOPT_MAX_OPTIONS * sizeof(u_char));

	ipopt_reg_proto(IPOPT_RR, IPOPT_RR_OFFSET);
	ipopt_reg_proto(IPOPT_TS, IPOPT_TS_OFFSET);
	ipopt_reg_proto(IPOPT_SECURITY, IPOPT_SEC_OFFSET);
	ipopt_reg_proto(IPOPT_LSRR, IPOPT_LSRR_OFFSET);
	ipopt_reg_proto(IPOPT_SSRR, IPOPT_SSRR_OFFSET);
	ipopt_reg_proto(IPOPT_RA, IPOPT_RA_OFFSET);

	/* init PCB chains */
	LIST_INIT(&ipoptcb_ralist);
	LIST_INIT(&ipoptcb_rrlist);
	LIST_INIT(&ipoptcb_tslist);
	LIST_INIT(&ipoptcb_lsrrlist);
	LIST_INIT(&ipoptcb_ssrrlist);
	LIST_INIT(&ipoptcb_seclist);
}


/* protocol registration: walk through ipoptsw[] to find out
 * a protocol's location. We then store the location into
 * the offset array, s.t. in the forwarding path, we can call the 
 * protocol input function (in ipoptsw[]) in one step.
 */
void
ipopt_reg_proto(int proto, int offset)
{ 
	struct protosw *pr;

	pr = pffindproto(PF_IPOPTION, proto, SOCK_RAW);
	if (pr == 0)
		ipopt_protox[offset] = 0;	/* default one */
	else
		ipopt_protox[offset] = pr - ipoptsw;
}


/* ipopt socket option processing: called by setsockopt() and
 * getsocketopt(). To set a socket option, we keep the req in both
 * ipopt_cfg[] and PCB itself.
 */
int
ipopt_ctloutput(struct socket *so, struct sockopt *sopt)
{
	struct ipoptpcb *ocb = sotoipoptcb(so);
	int error, val;

	if (ocb == 0)
		return EINVAL;
	if (sopt->sopt_level != IPPROTO_IP)
		return EINVAL;

	error = val = 0;

	switch (sopt->sopt_dir) {
	case SOPT_SET:
		error = sooptcopyin(sopt, &val, sizeof val, sizeof val);
		if (error)
			break;

		switch (sopt->sopt_name) {
		case IP_RECVIF:
			if (OCB_RECVIF_SET(ocb)) {
				error = EEXIST;
				break;
			}

			OCB_SET_RECVIF(ocb);
			break;

		case IPOPT_RECVLOCAL:
			if (OCB_RECVLOCAL_SET(ocb)) {
				error = EEXIST;
				break;
			}

			OCB_SET_RECVLOCAL(ocb);
			break;

		case IPOPT_RECVRA:
			if (ocb->ocb_proto.sp_protocol != IPOPT_RA) {
				error = EINVAL;
				break;
			}
			if (OCB_RA_SET(ocb)) {
				error = EEXIST;
				break;
			}

			ipoptstat.allra_count++;
			IPOPT_SET_ALLRA(&ipopt_cfg);
			OCB_SET_RA(ocb);
			break;

		case IPOPT_RECVRSVP:
			if (ocb->ocb_proto.sp_protocol != IPOPT_RA) {
				error = EINVAL;
				break;
			}
			if (OCB_RSVP_SET(ocb)) {
				error = EEXIST;
				break;
			}

			ipoptstat.rsvp_count++;
			IPOPT_SET_ARSVP(&ipopt_cfg);
			OCB_SET_RSVP(ocb);
			break;

		case IPOPT_RECVRTCP:

			if (ocb->ocb_proto.sp_protocol != IPOPT_RA) {
				error = EINVAL;
				break;
			}
			if (OCB_RTCP_SET(ocb)) {
				error = EEXIST;
				break;
			}

			ipoptstat.rtcp_count++;
			IPOPT_SET_ARTCP(&ipopt_cfg);
			OCB_SET_RTCP(ocb);
			break;

		default:
			error = ENOPROTOOPT;
			break;
		}
		break;
	
	case SOPT_GET:
		switch (sopt->sopt_name) {
		case IP_RECVIF:
			val = OCB_BIT(ocb, OCB_RECVIF);
			break;

		case IPOPT_RECVLOCAL:
			val = OCB_BIT(ocb, OCB_RECVLOCAL);
			break;

		case IPOPT_RECVRA:
			if (ocb->ocb_proto.sp_protocol != IPOPT_RA) {
				error = EINVAL;
				break;
			}
			val = OCB_BIT(ocb, OCB_RECVRA);
			break;

		case IPOPT_RECVRSVP:
			if (ocb->ocb_proto.sp_protocol != IPOPT_RA) {
				error = EINVAL;
				break;
			}
			val = OCB_BIT(ocb, OCB_RECVRSVP);
			break;

		case IPOPT_RECVRTCP:
			if (ocb->ocb_proto.sp_protocol != IPOPT_RA) {
				error = EINVAL;
				break;
			}
			val = OCB_BIT(ocb, OCB_RECVRTCP);
			break;

		default:
			error = ENOPROTOOPT;
			break;
		}

		if (!error)
			error = sooptcopyout(sopt, &val, sizeof val); 
		break;
	}
	return error;
}


/* functions below are used in pr_usrreqs to support socket-level operations.
 * Again check Steven's book for pr_usrreqs detail.
 */

/* disconnect the control block, release and disconnect the socket;
 */

static int
ipopt_abort(struct socket *so)
{
	struct ipoptpcb *ocb = sotoipoptcb(so);
	int s, error;

	if (ocb == 0)
		return EINVAL;

	s = splnet();
	ipopt_reset_cb(so);
	error = ipopt_free_cb(so);
	soisdisconnected(so);
	splx(s);

	return error;
}


/* allocate and insert a control block to the PCB chain, and get socket 
 * buffer space
 */

static int
ipopt_attach(struct socket *so, int proto, struct proc *p)
{
	struct ipoptpcb *ocb;
	int s, error;

	if (sotoipoptcb(so) != 0)
		return EISCONN;

/* BSD 3.3 and 4.0 are different
	if (p && (error = suser(p)) != 0)
	if (p && (error = suser(p->p_ucred, &p->p_acflag)) != 0)
		return error;
 */

	MALLOC(ocb, struct ipoptpcb *, sizeof *ocb, M_PCB, M_WAITOK);
	if (ocb == 0)
		return ENOBUFS;

	bzero(ocb, sizeof *ocb);

	s = splnet();
	so->so_pcb = (caddr_t)ocb;
	error = soreserve(so, ipopt_sendspace, ipopt_recvspace);
	if (error) {
		splx(s);
		free(ocb, M_PCB);
		return (error);
	}

	/* make sure the protocol os valid */
	error = ipopt_accept_proto(proto, ocb);
	if (error) {
		splx(s);
		free(ocb, M_PCB);
		return error;
	}

	/* init ocb */
	ocb->ocb_socket = so;
	ocb->ocb_proto.sp_family = so->so_proto->pr_domain->dom_family;
	ocb->ocb_proto.sp_protocol = proto;

	soisconnected(so);
	splx(s);

	return 0;
}


/* do nothing for raw 
 */

static int
ipopt_bind(struct socket *so, struct sockaddr *nam, struct proc *p)
{
	return EINVAL;
}

/* do nothing for raw 
 */

static int
ipopt_connect(struct socket *so, struct sockaddr *nam, struct proc *p)
{
	return EINVAL;
}


/* release the socket... just like ipopt_abort() 
 */

static int
ipopt_detach(struct socket *so)
{
	struct ipoptpcb *ocb = sotoipoptcb(so);
	int s, error;

	if (ocb == 0)
		return EINVAL;

	s = splnet();
	ipopt_reset_cb(so);
	error = ipopt_free_cb(so);
	splx(s);

	return error;
}


/* same as detach 
 */

static int
ipopt_disconnect(struct socket *so)
{
	struct ipoptpcb *ocb = sotoipoptcb(so);
	int s, error;

	if (ocb == 0)
		return EINVAL;

	s = splnet();
	ipopt_reset_cb(so);
	error = ipopt_free_cb(so);
	splx(s);

	return error;
}


/* for now, I don't support remote address setting 
 */
static int
ipopt_peeraddr(struct socket *so, struct sockaddr **nam)
{
	return EINVAL;
}


/* process writing to the socket: do nothing since the extnsion takes
 * care of receiving path only.
 */

static int
ipopt_send(struct socket *so, int flags, struct mbuf *m, struct sockaddr *nam,
	 struct mbuf *control, struct proc *p)
{
	int s;

	s = splnet();

	if (m != NULL)
		m_freem(m);

	splx(s);

	return EINVAL;
}


/* do nothing for I have no connection to shutdown
 */

static int
ipopt_shutdown(struct socket *so)
{
	struct ipoptpcb *ocb = sotoipoptcb(so);
	int s;

	if (ocb == 0)
		return EINVAL;

	s = splnet();
	socantsendmore(so);
	splx(s);

	return 0;
}


/* don't think I need socket address for this option (that's why there is
 * no such field in the ipoptpcb structure.). If have to have it someday,
 * just return the address to <nam>, something like:
 *	...
 *	if (ocb->ocb_addr == 0)
 *		return EINVAL;
 *	*nam = dup_sockaddr(ocb->ocb_addr, 1);
 *	return 0;
 */

static int
ipopt_sockaddr(struct socket *so, struct sockaddr **nam)
{
	struct ipoptpcb *ocb = sotoipoptcb(so);

	if (ocb == 0)
		return EINVAL;

	return EOPNOTSUPP;
}


struct pr_usrreqs	ipopt_usrreqs = {
	ipopt_abort, pru_accept_notsupp, ipopt_attach, ipopt_bind, 
	ipopt_connect, pru_connect2_notsupp, pru_control_notsupp, 
	ipopt_detach, ipopt_disconnect, pru_listen_notsupp, ipopt_peeraddr, 
	pru_rcvd_notsupp, pru_rcvoob_notsupp, ipopt_send, pru_sense_null, 
	ipopt_shutdown, ipopt_sockaddr, sosend, soreceive, sopoll
};


/* utility functions:
 */

/* reset the system parameter 
 */

void
ipopt_reset_cb(struct socket *so)
{
	struct ipoptpcb *ocb = sotoipoptcb(so);

	switch(ocb->ocb_proto.sp_protocol) {
	case IPOPT_RR:
		ipoptstat.rr_count--;
		if (!ipoptstat.rr_count)
			IPOPT_RESET_RR(&ipopt_cfg);
		break;
	case IPOPT_TS:
		ipoptstat.ts_count--;
		if (!ipoptstat.ts_count)
			IPOPT_RESET_TS(&ipopt_cfg);
		break;
	case IPOPT_SECURITY:
		ipoptstat.sec_count--;
		if (!ipoptstat.sec_count)
			IPOPT_RESET_SEC(&ipopt_cfg);
		break;
	case IPOPT_LSRR:
		ipoptstat.lsrr_count--;
		if (!ipoptstat.lsrr_count)
			IPOPT_RESET_LSRR(&ipopt_cfg);
		break;
	case IPOPT_SSRR:
		ipoptstat.ssrr_count--;
		if (!ipoptstat.ssrr_count)
			IPOPT_RESET_SSRR(&ipopt_cfg);
		break;
	case IPOPT_RA:
		ipoptstat.ra_count--;
		if (!ipoptstat.ra_count) {
			IPOPT_RESET_RA(&ipopt_cfg);
			IPOPT_RESET_ALLRA(&ipopt_cfg);
			IPOPT_RESET_ARTCP(&ipopt_cfg);
			IPOPT_RESET_ARSVP(&ipopt_cfg);
			break;
		}

		if (OCB_RA_SET(ocb)) {
			ipoptstat.allra_count--;
			if (!ipoptstat.allra_count)
				IPOPT_RESET_ALLRA(&ipopt_cfg);
		}

		if (OCB_RTCP_SET(ocb)) {
			ipoptstat.rtcp_count--;
			if (!ipoptstat.rtcp_count)
				IPOPT_RESET_ARTCP(&ipopt_cfg);
		}

		if (OCB_RSVP_SET(ocb)) {
			ipoptstat.rsvp_count--;
			if (!ipoptstat.rsvp_count)
				IPOPT_RESET_ARSVP(&ipopt_cfg);
		}

		break;
	}
}


/* remove the control block from the chain list and free the memory
 */

int
ipopt_free_cb(struct socket *so)
{
	struct ipoptpcb *ocb = sotoipoptcb(so);

	if (ocb == 0)
		return EINVAL;
	if (ocb->ocb_socket != so)	/* huh? */
		return EINVAL;

	so->so_pcb = 0;
	sofree(so);
	LIST_REMOVE(ocb, list);
	free((caddr_t)(ocb), M_PCB);
	return 0;
}


/* called from attach func to set the user req in ipopt_cfg[].
 * The code is kinda stupid...
 */

int
ipopt_accept_proto(int proto, struct ipoptpcb *ocb)
{
	int error=0;

	switch(proto) {
	case IPOPT_RR:
		ipoptstat.rr_count++;
		IPOPT_SET_RR(&ipopt_cfg);
		LIST_INSERT_HEAD(&ipoptcb_rrlist, ocb, list);
		break;

	case IPOPT_TS:
		ipoptstat.ts_count++;
		IPOPT_SET_TS(&ipopt_cfg);
		LIST_INSERT_HEAD(&ipoptcb_tslist, ocb, list);
		break;

	case IPOPT_SECURITY:
		ipoptstat.sec_count++;
		IPOPT_SET_SEC(&ipopt_cfg);
		LIST_INSERT_HEAD(&ipoptcb_seclist, ocb, list);
		break;

	case IPOPT_LSRR:
		ipoptstat.lsrr_count++;
		IPOPT_SET_LSRR(&ipopt_cfg);
		LIST_INSERT_HEAD(&ipoptcb_lsrrlist, ocb, list);
		break;

	case IPOPT_SSRR:
		ipoptstat.ssrr_count++;
		IPOPT_SET_SSRR(&ipopt_cfg);
		LIST_INSERT_HEAD(&ipoptcb_ssrrlist, ocb, list);
		break;

	case IPOPT_RA:
		ipoptstat.ra_count++;
		IPOPT_SET_RA(&ipopt_cfg);
		LIST_INSERT_HEAD(&ipoptcb_ralist, ocb, list);
		break;

	default:
		error = EPROTONOSUPPORT;
	}
	return error;
}
