/*
 * Copyright (c) 2000, Ping Pan and Henning Schulzrinne
 *	All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Initially written by Ping Pan, Columbia University/Bell Labs, February 2000.
 * We acknowledge Bell Labs for support. Please forward bug fixes,enhancements 
 * and questions to pingpan@cs.columbia.edu.
 */

/* Both ipopt_map() and ipopt_input() are directly called from ip_dooptions()
 * inside the ip_input().
 */

#include <stddef.h>

#define _IP_VHL
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/sysctl.h>
#include <sys/proc.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/domain.h>
#include <sys/protosw.h>
#include <net/raw_cb.h>
#include <net/if.h>
#include <net/if_var.h>
#include <net/if_dl.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netipopt/ipopt_var.h>


static struct	sockaddr_in ipoptsrc = {sizeof(ipoptsrc), AF_IPOPTION};

/* Input:
 * 	opt:		option value
 *	payload:	pointer to the option field
 *	ip		pointer to the IP header
 *
 * The routine reads the option and sets the corresponding bit in
 * <ipopt_pkt>. For Router-Alert option, we check the alert value also.
 * The <ipopt_pkt> is a bitmap for received packet's IP options,
 * and is used in ipopt_input() for fast lookup.
 */
 
void
ipopt_map(register int opt, u_char *payload, struct ip *ip)
{
	bzero((char *)&ipopt_pkt, sizeof(struct ipopt_mask));

	switch (opt) {
	case IPOPT_RR:
		IPOPT_SET_RR(&ipopt_pkt);
		break;
	case IPOPT_TS:
		IPOPT_SET_TS(&ipopt_pkt);
		break;
	case IPOPT_SECURITY:
		IPOPT_SET_SEC(&ipopt_pkt);
		break;
	case IPOPT_LSRR:
		IPOPT_SET_LSRR(&ipopt_pkt);
		break;
	case IPOPT_SSRR:
		IPOPT_SET_SSRR(&ipopt_pkt);
		break;
	case IPOPT_RA: {
		register struct ip_routeralert *alert;
		register u_short ra_val;

		alert = (struct ip_routeralert *)payload;
		ra_val = htons(alert->ipa_val);

		IPOPT_SET_RA(&ipopt_pkt);

		if ((ra_val == IPOPT_RA_EXAM) && (ip->ip_p == IPPROTO_RSVP)) {
			/* RFC2113 */
			IPOPT_SET_ARSVP(&ipopt_pkt);
		}
		else if (ra_val == IPOPT_RA_RTCP) {
			/* Pan-Schulzrinne Router-Alert draft */
			IPOPT_SET_ARTCP(&ipopt_pkt);
		}

		break;
	}
	default:
		break;
	}
	return;
}


/* This routine first checks whether the received packet is wanted
 * by the user. (The packet's IP option data is in <ipopt_pkt>;
 * the user's requests are in <ipopt_cfg>.) For Router Alert, we do
 * more checkings. If there is at least one match, copy the packet to a mbuf,
 * and call the pre-registered upcall function in ipoptsw[] to
 * send the copy to the process. We are trying to make the path as fast
 * as possible, given this routine is called inside ip_input().
 *
 * I assume an IP header can have more than one option, thus a packet
 * may need to be copied to multiple processes. Copying operation can
 * be very bad for the performance. In real life, I have never seen a packet 
 * with more than one option. If so, don't copy to another mbuf here,
 * just send the current one.
 *
 * Return:
 *	-1	the packet's been processed and the buffer's been freed.
 *	0	didn't do anything; pass it back to the forwarding loop 
 */

int
ipopt_input(struct mbuf *m, int flag)
{
	register struct ip *ip;
	register struct mbuf *n;
	register u_long opt_list, hlen, i, j;

	ip = mtod(m, struct ip *);
	hlen = IP_VHL_HL(ip->ip_vhl) << 2;

	/* a regular packet */
	if (hlen == sizeof (struct ip))
		return 0;

	/* do we need to capture? */
	opt_list = (ipopt_pkt.flag & ipopt_cfg.flag);
	if (opt_list == 0)		/* no match */
		return 0;

	/* is it the router-alert pkt's that I want ? */
	if ((opt_list & IPOPT_RA_MASK) && !IPOPT_ALLRA_SET(&ipopt_cfg)) {
		if (!(ipopt_pkt.alert_flag & ipopt_cfg.alert_flag))
				return 0;
	}

	for (i=1, j=0; j < IPOPT_MAX_OPTIONS; i <<=1,j++) {

		/* at this point, we know that at least one message 
		 * is heading up, so we do the copy.
		 */
		if (opt_list & i) {	
			n = m_copy(m, 0, (int)M_COPYALL);
			if (n)
				(*ipoptsw[ipopt_protox[j]].pr_input)(n, flag);
			else
				ipoptstat.no_space++;
		}
	}

	/* if the packet is local, leave it alone
	 */
	if (flag == IPOPT_LOCAL)
		return 0;

	m_freem(m);
	return -1;

}


/* 
 * Below are all the functions called by ipopt_input().
 * The function registration takes place in ipopt_init().
 */

/* The routine is called when the incoming option type is 
 * new or not supported. ... registered because pffindproto() returns
 * zero in ipopt_reg_proto().
 */
void
null_input(struct mbuf *m, int flag)
{
	ipoptstat.null_input++;
	m_freem(m);
	return;
}

/* record route input */
void
rr_input(struct mbuf *m, int flag)
{
	ipoptstat.rr_input++;
	ipopt_proto_input(m, flag, IPOPT_RR, &ipoptcb_rrlist);
}

/* timestamp input */
void
ts_input(struct mbuf *m, int flag)
{
	ipoptstat.ts_input++;
	ipopt_proto_input(m, flag, IPOPT_TS, &ipoptcb_tslist);
}

/* security input */
void
security_input(struct mbuf *m, int flag)
{
	ipoptstat.sec_input++;
	ipopt_proto_input(m, flag, IPOPT_SECURITY, &ipoptcb_seclist);
}

/* loose soured routing input */
void
lsrr_input(struct mbuf *m, int flag)
{
	ipoptstat.lsrr_input++;
	ipopt_proto_input(m, flag, IPOPT_LSRR, &ipoptcb_lsrrlist);
}

/* strict sourced routing input */
void
ssrr_input(struct mbuf *m, int flag)
{
	ipoptstat.ssrr_input++;
	ipopt_proto_input(m, flag, IPOPT_SSRR, &ipoptcb_ssrrlist);
}


/* router alert input:
 * Walk through the PCB chain and send up the Router-Alert message. 
 * For each RA's PCB, we first make sure that if it's to capture all 
 * incoming RA's, or to capture only specific RA type. Always check if
 * we need to append control data ahead of the message.
 */

void
ra_input(struct mbuf *m, int flg)
{
	register struct ip *ip = mtod(m, struct ip *);
	register struct ipoptpcb *ocb, *last;
	struct mbuf *opts;

	opts = 0;
	last = 0;
	ipoptstat.ra_input++;
	ipoptsrc.sin_addr = ip->ip_src;

	LIST_FOREACH(ocb, &ipoptcb_ralist, list) {

		if (ocb->ocb_proto.sp_protocol != IPOPT_RA)
			continue;

		/* if this OCB is for a spec application */
		if (!OCB_RA_SET(ocb)) {
			/* skip if we know there is an OCB being created
			 * to capture all RA's,
			 */
			if (IPOPT_ALLRA_SET(&ipopt_cfg))
				continue;

			if (IPOPT_ARSVP_SET(&ipopt_pkt) && !OCB_RSVP_SET(ocb))
				continue;

			if (IPOPT_ARTCP_SET(&ipopt_pkt) && !OCB_RTCP_SET(ocb))
				continue;
		}

		if (last) {
			struct mbuf *n;
			n = m_copy(m, 0, (int)M_COPYALL);
			if (n) {
				ipopt_appenctl(last, &opts, ip, n, flg);

				if (sbappendaddr(&last->ocb_socket->so_rcv, 
				    (struct sockaddr *)&ipoptsrc, n,
				    opts) == 0)  {
					ipoptstat.lost_input++;
					m_freem(n);
					if (opts) 
						m_freem(opts);
				}
				else {
					sorwakeup(last->ocb_socket);
				}
			}
		}
		last = ocb;
	}
	if (last) {
		ipopt_appenctl(last, &opts, ip, m, flg);

		if (sbappendaddr(&last->ocb_socket->so_rcv, 
		    (struct sockaddr *)&ipoptsrc, m, opts) == 0) {
			ipoptstat.lost_input++;
			m_freem(m);
			if (opts)
				m_freem(opts);
		}
		else {
			sorwakeup(last->ocb_socket);
		}
	} else {
		m_freem(m);
		if (opts)
			m_freem(opts);
	}
}


/* 
 * generic IP Option input
 * Walk through the chain, and send up the corresponding message.
 */
void
ipopt_proto_input(m, flg, proto, pcb_list)
	register struct mbuf *m;
	register int flg;
	register int proto;
	register struct ipoptcb_list_head *pcb_list;
{
	register struct ip *ip = mtod(m, struct ip *);
	register struct ipoptpcb *ocb, *last;
	struct mbuf *opts;

	opts = 0;
	last = NULL;
	ipoptsrc.sin_addr = ip->ip_src;

	LIST_FOREACH(ocb, pcb_list, list) {
		if (ocb->ocb_proto.sp_protocol != proto)
			continue;

		if (last) {
			struct mbuf *n;
			n = m_copy(m, 0, (int)M_COPYALL);
			if (n) {
				ipopt_appenctl(last, &opts, ip, n, flg);

				if (sbappendaddr(&last->ocb_socket->so_rcv, 
				    (struct sockaddr *)&ipoptsrc, n,
				    opts) == 0)  {
					ipoptstat.lost_input++;
					m_freem(n);
					if (opts) 
						m_freem(opts);
				}
				else {
					sorwakeup(last->ocb_socket);
				}
			}
		}
		last = ocb;
	}
	if (last) {
		ipopt_appenctl(last, &opts, ip, m, flg);

		if (sbappendaddr(&last->ocb_socket->so_rcv, 
		    (struct sockaddr *)&ipoptsrc, m, opts) == 0) {
			ipoptstat.lost_input++;
			m_freem(m);
			if (opts)
				m_freem(opts);
		}
		else {
			sorwakeup(last->ocb_socket);
		}
	} else {
		m_freem(m);
		if (opts)
			m_freem(opts);
	}
}


/* By default, we always copy the local flag. Given we may not want to 
 * run extensive route lookup in the application, so we cheat here.
 *
 * The RECVIF was originally written by Bill Fenner for UDP sockets. 
 * It is to insert ingress interface index ahead of the message.
 * This can be quite helpful for protocol processing.
 * (copied from ip_savecontrol())
 */

void
ipopt_appenctl(ocb, mp, ip, m, flg)
	register struct ipoptpcb *ocb; 
	register struct mbuf **mp;
	register struct ip *ip;
	register struct mbuf *m;
	int flg;
{
	struct ifnet *ifp;
	struct sdlbuf {
		struct sockaddr_dl sdl;
		u_char	pad[32];
	} sdlbuf;
	struct sockaddr_dl *sdp;
	struct sockaddr_dl *sdl2 = &sdlbuf.sdl;
	int n;

	/* copy the local flag 
	 */
	*mp = sbcreatecontrol((caddr_t)&flg, sizeof(flg), 
		IPOPT_RECVLOCAL, IPPROTO_IP);
	if (*mp)
		mp = &(*mp)->m_next;

	/* get the interface index also ? */
	if (!OCB_RECVIF_SET(ocb))
		return;

	ifp = (struct ifnet *)(m->m_pkthdr.rcvif);
	if (ifp && ifp->if_index && (ifp->if_index <= if_index)) {

		n = ifp->if_index;
		sdp = (struct sockaddr_dl *)(ifnet_addrs[n-1]->ifa_addr);

		/* Change our mind and don't try copy. */
		if ((sdp->sdl_family != AF_LINK)
		|| (sdp->sdl_len > sizeof(sdlbuf))) {
			goto makedummy;
		}
		bcopy(sdp, sdl2, sdp->sdl_len);
	}
	else {
makedummy:	
		sdl2->sdl_len = offsetof(struct sockaddr_dl, sdl_data[0]);
		sdl2->sdl_family = AF_LINK;
		sdl2->sdl_index = 0;
		sdl2->sdl_nlen = sdl2->sdl_alen = sdl2->sdl_slen = 0;
	}

	*mp = sbcreatecontrol((caddr_t) sdl2, sdl2->sdl_len,
		IP_RECVIF, IPPROTO_IP);
	if (*mp)
		mp = &(*mp)->m_next;
}

