package edu.columbia.preju.prediction;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import edu.columbia.preju.core.Effect;
import edu.columbia.preju.core.Importance;
import edu.columbia.preju.prediction.classifier.ClassInfo;
import edu.columbia.preju.prediction.classifier.ClassifierModelInfo;
import ob.core.Feature;
import ob.core.Value;
import ob.util.Utils;

/**
 * @author Or
 *
 */
public class ClassifierPrediction implements Prediction {

	private String _subjectDomainName;
	private int _classIndex;
	private Value _predictedClass;
	private Map<Feature, Value> _featureValues = new HashMap<Feature, Value>();
	private ClassifierModelInfo _modelInfo;
	
	public ClassifierPrediction(String subjectDomainName, int classIndex, Value predictedClass, Map<Feature, Value> featureValues, ClassifierModelInfo modelInfo) {
		_subjectDomainName = subjectDomainName;
		_classIndex = classIndex;
		_predictedClass = predictedClass;
		_modelInfo = modelInfo;
		_featureValues.putAll(featureValues);
		_featureValues.put(ClassInfo.PRIOR_FEATURE, _modelInfo.getPrior(_classIndex));
	}

	
	@Override
	public String getSubjectDomainName() {
		return _subjectDomainName;
	}

	@Override
	public Collection<Feature> getFeatures() {
		return _featureValues.keySet();
	}

	@Override
	public Effect getEffect(Feature feature) {
		return _modelInfo.getEffect(feature, _classIndex, _featureValues);
	}

	@Override
	public Importance getImportance(Feature feature) {
		return _modelInfo.getImportance(feature, _classIndex);
	}

	@Override
	public String getModelName() {
		return _modelInfo.getModelName();
	}

	@Override
	public Value getPredictedClass() {
		return _predictedClass;
	}

	@Override
	public void exportToXml(String fileName) {
		List<String> lines = new ArrayList<String>();
		
		lines.add("<prediction domain=\"" + _subjectDomainName + "\">");
		lines.add("\t" + "<classification classIndex=\"" + _classIndex + "\" className=\"" + _predictedClass.getString() + "\">");
		for (Feature feature : _featureValues.keySet()) {
			lines.add("\t\t" + "<feature name=\"" + feature.getName() + "\" value=\"" + _featureValues.get(feature).getNumber() + "\"/>");
		}
		lines.add("\t" + "</classification>");
		
		lines.add("\t" + "<model name=\"" + _modelInfo.getModelName() + "\">");

		lines.add("\t\t" + "<priors>");
		for (int i=0; i<_modelInfo.getNumClasses(); i++) {
			if (_modelInfo.getPrior(i) != null) {
				double prior = _modelInfo.getPrior(i).getNumber();
				lines.add("\t\t\t" + "<prior classIndex=\"" + i + "\" value=\"" + prior + "\"/>");
			}
		}
		lines.add("\t\t" + "</priors>");

		lines.add("\t\t" + "<importances>");
		for (int i=0; i<_modelInfo.getNumClasses(); i++) {
			for (Feature feature : _modelInfo.getFeatures()) {
				lines.add("\t\t\t" + "<importance classIndex=\"" + i + "\" feature=\"" + feature.getName() + "\" value=\"" + _modelInfo.getImportance(feature, i).getDouble() + "\"/>");
			}
		}
		lines.add("\t\t" + "</importances>");

		lines.add("\t\t" + "<effects>");
		for (int i=0; i<_modelInfo.getNumClasses(); i++) {
			for (Feature feature : _modelInfo.getFeatures()) {
				lines.add("\t\t\t" + "<effect classIndex=\"" + i + "\" feature=\"" + feature.getName() + "\" value=\"" + _modelInfo.getEffect(feature, i, _featureValues).getDouble() + "\"/>");
			}
		}
		lines.add("\t\t" + "</effects>");
		
		lines.add("\t" + "</model>");
		
		lines.add("</prediction>");
		
		Utils.writeLinesToFile(fileName, lines);
	}

	@Override
	public void exportSubjectDomainPackage() {
		File dir = new File("domain", _subjectDomainName);
		dir.mkdir();
		
		File messagesDir = new File(dir, "messages");
		messagesDir.mkdir();
		Utils.writeLinesToFile(new File(messagesDir, "core").getAbsolutePath(), new ArrayList<String>());
		Utils.writeLinesToFile(new File(messagesDir, "definitional").getAbsolutePath(), new ArrayList<String>());
		Utils.writeLinesToFile(new File(messagesDir, "taxonomy").getAbsolutePath(), new ArrayList<String>());
		
		File registryDir = new File(dir, "registry");
		registryDir.mkdir();
		Utils.writeLinesToFile(new File(registryDir, "entities").getAbsolutePath(), makeEntityLines());
		Utils.writeLinesToFile(new File(registryDir, "extra_entities").getAbsolutePath(), new ArrayList<String>());
		Utils.writeLinesToFile(new File(registryDir, "stt_discourse").getAbsolutePath(), new ArrayList<String>());
		Utils.writeLinesToFile(new File(registryDir, "stts").getAbsolutePath(), new ArrayList<String>());
	}

	private Collection<String> makeEntityLines() {
		List<String> lines = new ArrayList<String>();
		int i=0;
		for (Feature feature : _featureValues.keySet()) {
			lines.add("de" + (++i) + "\t" + "feature" + "\t" + feature.getName());
		}
		return lines;
	}

	
}
