package edu.columbia.preju.generator.htc;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;

import edu.columbia.ob.gen.core.Paragraph;
import edu.columbia.ob.gen.core.ParagraphImpl;
import edu.columbia.ob.gen.core.SemanticUnit;
import edu.columbia.ob.gen.core.stub.SemanticUnitStub;
import edu.columbia.preju.core.NarrativeRole;
import edu.columbia.preju.generator.JustificationNarrative;
import ob.core.Feature;
import ob.util.Utils;

/**
 * @author Or
 *
 */
public class HandCraftedWithDomainPrejuParagraphGenerator {

	private StringDecorator _featureEmphasizer;
	
	public HandCraftedWithDomainPrejuParagraphGenerator(StringDecorator featureEmphasizer) {
		_featureEmphasizer = featureEmphasizer;
	}

	public Collection<Paragraph> getParagraphs(JustificationNarrative narrative) {
		Collection<Paragraph> paragraphs = new ArrayList<Paragraph>();
		
		List<SemanticUnit> evidenceSus = makeEvidenceSus(narrative);
		List<SemanticUnit> missingEvidenceSus = makeMissingEvidenceSus(narrative);
		List<SemanticUnit> counterEvidenceSus = makeCounterEvidenceSus(narrative);
		List<SemanticUnit> missingCounterEvidenceSus = makeMissingCounterEvidenceSus(narrative);
		
		int r = new Random().nextInt(3);
		
		if (r==0) {
			if (evidenceSus != null) paragraphs.add(new ParagraphImpl(evidenceSus));
			if (missingEvidenceSus != null) paragraphs.add(new ParagraphImpl(missingEvidenceSus));
			if (counterEvidenceSus != null) paragraphs.add(new ParagraphImpl(counterEvidenceSus));
			if (missingCounterEvidenceSus != null) paragraphs.add(new ParagraphImpl(missingCounterEvidenceSus));
		}
		else if (r==1) {
			ParagraphImpl evidenceParagraph = new ParagraphImpl();
			if (evidenceSus != null) evidenceParagraph.addSemanticUnits(evidenceSus);
			if (missingEvidenceSus != null) evidenceParagraph.addSemanticUnits(missingEvidenceSus);
			paragraphs.add(evidenceParagraph);
			
			ParagraphImpl counterEvidenceParagraph = new ParagraphImpl();
			if (counterEvidenceSus != null) counterEvidenceParagraph.addSemanticUnits(counterEvidenceSus);
			if (missingCounterEvidenceSus != null) counterEvidenceParagraph.addSemanticUnits(missingCounterEvidenceSus);
			paragraphs.add(counterEvidenceParagraph);
		}
		else if (r==2) {
			ParagraphImpl evidenceParagraph = new ParagraphImpl();
			if (evidenceSus != null) evidenceParagraph.addSemanticUnits(evidenceSus);
			if (counterEvidenceSus != null) evidenceParagraph.addSemanticUnit(counterEvidenceSus.get(0));  // note here we're only taking the base counter-evidence SU, without the elaboration
			paragraphs.add(evidenceParagraph);
			
			ParagraphImpl missingEvidenceParagraph = new ParagraphImpl();
			if (missingEvidenceSus != null) missingEvidenceParagraph.addSemanticUnits(missingEvidenceSus);
			paragraphs.add(missingEvidenceParagraph);
			
			ParagraphImpl missingCounterEvidenceParagraph = new ParagraphImpl();
			if (missingCounterEvidenceSus != null) missingCounterEvidenceParagraph.addSemanticUnits(missingCounterEvidenceSus);
			paragraphs.add(missingCounterEvidenceParagraph);

		}

		return paragraphs;
	}

	private List<SemanticUnit> makeMissingCounterEvidenceSus(final JustificationNarrative narrative) {
		List<Feature> missingCounterEvidenceFeatures = getFeatures(narrative.getKeyFeatures(), new FeatureSelector() {
			public boolean select(Feature feature) {
				return narrative.getNarrativeRole(feature).equals(NarrativeRole.MissingCounterEvidence);
			}
		});
		
		if (missingCounterEvidenceFeatures.isEmpty()) return null;
		
		String featureString = makeFeaturesString(missingCounterEvidenceFeatures);
		
		List<SemanticUnit> sus = new ArrayList<SemanticUnit>();
		
		SemanticUnit su = new SemanticUnitStub(
				"#normally , #there_would_be #strong counter-evidence in " + featureString + " , but #it_is_missing #in_this_case",
				"#normally , #there_would_be #strong counter-evidence in " + featureString + " , but #in_this_case #it_is_missing",
				"although #normally #there_would_be #strong counter-evidence in " + featureString + " , #in_this_case #it_is_missing",
				"although #normally #there_would_be #strong counter-evidence in " + featureString + " , #it_is_missing #in_this_case"
				); 
		
		sus.add(su);
		if (missingCounterEvidenceFeatures.size() == 1 && new Random().nextDouble() >= 0.87) {
			SemanticUnit groundingSu = getGroundingSu(missingCounterEvidenceFeatures.get(0));
			if (groundingSu != null) {
				sus.add(groundingSu);
				System.out.println("grounding for " + CurrentPredictionProperties.getPredictedObjectName());
			}
		}
		if (missingCounterEvidenceFeatures.size() == 2 && new Random().nextDouble() >= 0.95) {
			SemanticUnit groundingSu = getGroundingSu(missingCounterEvidenceFeatures.get(1));
			if (groundingSu != null) {
				sus.add(groundingSu);
				System.out.println("grounding for " + CurrentPredictionProperties.getPredictedObjectName());
			}
		}

		return sus;
	}

	private List<SemanticUnit> makeMissingEvidenceSus(final JustificationNarrative narrative) {
		List<Feature> missingEvidenceFeatures = getFeatures(narrative.getKeyFeatures(), new FeatureSelector() {
			public boolean select(Feature feature) {
				return narrative.getNarrativeRole(feature).equals(NarrativeRole.MissingEvidence);
			}
		});
		
		if (missingEvidenceFeatures.isEmpty()) return null;

		String featureString = makeFeaturesString(missingEvidenceFeatures);
		
		List<SemanticUnit> sus = new ArrayList<SemanticUnit>();
		
		SemanticUnit su = new SemanticUnitStub(
				"#normally , #there_would_be #strong evidence in " + featureString + " , but #it_is_missing #in_this_case",
				"#normally , #there_would_be #strong evidence in " + featureString + " , but #in_this_case #it_is_missing",
				"although #normally #there_would_be #strong evidence in " + featureString + " , #in_this_case #it_is_missing",
				"although #normally #there_would_be #strong evidence in " + featureString + " , #it_is_missing #in_this_case"
				); 
		
		sus.add(su);
		if (missingEvidenceFeatures.size() == 1 && new Random().nextDouble() >= 0.66) {
			SemanticUnit groundingSu = getGroundingSu(missingEvidenceFeatures.get(0));
			if (groundingSu != null) {
				sus.add(groundingSu);
				System.out.println("grounding for " + CurrentPredictionProperties.getPredictedObjectName());
			}
		}
		if (missingEvidenceFeatures.size() == 2 && new Random().nextDouble() >= 0.9) {
			SemanticUnit groundingSu = getGroundingSu(missingEvidenceFeatures.get(1));
			if (groundingSu != null) {
				sus.add(groundingSu);
				System.out.println("grounding for " + CurrentPredictionProperties.getPredictedObjectName());
			}
		}

		return sus;
	}

	private List<SemanticUnit> makeCounterEvidenceSus(final JustificationNarrative narrative) {
		List<Feature> counterEvidenceFeatures = getFeatures(narrative.getKeyFeatures(), new FeatureSelector() {
			public boolean select(Feature feature) {
				return narrative.getNarrativeRole(feature).isCounterEvidenceRole();
			}
		});
		
		List<Feature> exceptionalCounterEvidenceFeatures = getFeatures(narrative.getKeyFeatures(), new FeatureSelector() {
			public boolean select(Feature feature) {
				return narrative.getNarrativeRole(feature).equals(NarrativeRole.ExceptionalCounterEvidence);
			}
		});

		List<Feature> nonExceptionalCounterEvidenceFeatures = getFeatures(narrative.getKeyFeatures(), new FeatureSelector() {
			public boolean select(Feature feature) {
				return narrative.getNarrativeRole(feature).isCounterEvidenceRole() && ! narrative.getNarrativeRole(feature).equals(NarrativeRole.ExceptionalCounterEvidence);
			}
		});
		

		if (counterEvidenceFeatures.isEmpty()) return null;
		
		String featureString = makeFeaturesString(counterEvidenceFeatures);
		
		List<SemanticUnit> sus = new ArrayList<SemanticUnit>();
		
		SemanticUnit su = new SemanticUnitStub(
				"#strong counter-evidence #exists in " + featureString,
				"#there_is #strong counter-evidence in " + featureString
				); 
		
		sus.add(su);
		sus.add(makeNonMissingParagraphElaboration(counterEvidenceFeatures, exceptionalCounterEvidenceFeatures, nonExceptionalCounterEvidenceFeatures));
		if (counterEvidenceFeatures.size() == 1 && new Random().nextDouble() >= 0.75) {
			SemanticUnit groundingSu = getGroundingSu(counterEvidenceFeatures.get(0));
			if (groundingSu != null) {
				sus.add(groundingSu);
				System.out.println("grounding for " + CurrentPredictionProperties.getPredictedObjectName());
			}
		}
		if (counterEvidenceFeatures.size() == 2 && new Random().nextDouble() >= 0.85) {
			SemanticUnit groundingSu = getGroundingSu(counterEvidenceFeatures.get(1));
			if (groundingSu != null) {
				sus.add(groundingSu);
				System.out.println("grounding for " + CurrentPredictionProperties.getPredictedObjectName());
			}
		}

		return sus;
	}

	private List<SemanticUnit> makeEvidenceSus(final JustificationNarrative narrative) {
		List<Feature> evidenceFeatures = getFeatures(narrative.getKeyFeatures(), new FeatureSelector() {
			public boolean select(Feature feature) {
				return narrative.getNarrativeRole(feature).isEvidenceRole();
			}
		});
		
		List<Feature> exceptionalEvidenceFeatures = getFeatures(narrative.getKeyFeatures(), new FeatureSelector() {
			public boolean select(Feature feature) {
				return narrative.getNarrativeRole(feature).equals(NarrativeRole.ExceptionalEvidence);
			}
		});

		List<Feature> nonExceptionalEvidenceFeatures = getFeatures(narrative.getKeyFeatures(), new FeatureSelector() {
			public boolean select(Feature feature) {
				return narrative.getNarrativeRole(feature).isEvidenceRole() && ! narrative.getNarrativeRole(feature).equals(NarrativeRole.ExceptionalEvidence);
			}
		});
		

		if (evidenceFeatures.isEmpty()) return null;
		
		String featureString = makeFeaturesString(evidenceFeatures);

		List<SemanticUnit> sus = new ArrayList<SemanticUnit>();
		
		SemanticUnit su = new SemanticUnitStub(
				"the #main evidence for the #prediction is in " + featureString,
				"the #prediction was #decided #mostly #because_of the evidence in " + featureString,
				featureString + " " + (evidenceFeatures.size() == 1 ? "#constitutes" : "#constitute") + " the #main evidence for the #prediction",
				featureString + " " + (evidenceFeatures.size() == 1 ? "is" : "are") + " the #main evidence #because_of which the #prediction was #decided",
				"the evidence in " + featureString + " " + (evidenceFeatures.size() == 1 ? "#constitutes" : "#constitute") + " the #main #reason for the #prediction"
				); 
		
		sus.add(su);
		sus.add(makeNonMissingParagraphElaboration(evidenceFeatures, exceptionalEvidenceFeatures, nonExceptionalEvidenceFeatures));
		if (evidenceFeatures.size() == 1 && new Random().nextDouble() >= 0.5) {
			SemanticUnit groundingSu = getGroundingSu(evidenceFeatures.get(0));
			if (groundingSu != null) {
				sus.add(groundingSu);
				System.out.println("grounding for " + CurrentPredictionProperties.getPredictedObjectName());
			}
		}
		if (evidenceFeatures.size() == 2 && new Random().nextDouble() >= 0.75) {
			SemanticUnit groundingSu = getGroundingSu(evidenceFeatures.get(1));
			if (groundingSu != null) {
				sus.add(groundingSu);
				System.out.println("grounding for " + CurrentPredictionProperties.getPredictedObjectName());
			}
		}
		
		return sus;
	}

	private SemanticUnit getGroundingSu(Feature feature) {
		Map<String, List<String>> map = getGroundingSuMap();	

		List<String> sus = map.get(feature);
		if (sus == null) return null;
		return new SemanticUnitStub(sus.toArray(new String[sus.size()]));
	}


	private static Map<String, List<String>> getGroundingSuMap() {
		Map<String, List<String>> map = new HashMap<String, List<String>>();
		for (File file : new File("working/grounding").listFiles()) {
			map.put(file.getName(), Utils.readLines(file.getAbsolutePath()));
		}
		return map;
	}

	private SemanticUnit makeNonMissingParagraphElaboration(Collection<Feature> evidenceFeatures, Collection<Feature> exceptionalEvidenceFeatures, Collection<Feature> nonExceptionalEvidenceFeatures) {
		
		boolean one = evidenceFeatures.size() == 1;
		boolean oneE = exceptionalEvidenceFeatures.size() == 1;
		boolean oneN = nonExceptionalEvidenceFeatures.size() == 1;
		boolean allExceptional = evidenceFeatures.size() == exceptionalEvidenceFeatures.size();
		boolean hasExceptional = ! exceptionalEvidenceFeatures.isEmpty();
		
		String exceptionalEvidenceFeaturesString = makeFeaturesString(exceptionalEvidenceFeatures);
		String nonExceptionalEvidenceFeaturesString = makeFeaturesString(nonExceptionalEvidenceFeatures);
		if (allExceptional) {
			return new SemanticUnitStub("this is #unusual , #because " + (one ? "it is" : "these are") + " not #normally " + (one ? "a" : "") + " #strong indicator" + (one ? "" : "s"));
		}
		else if (hasExceptional) {
			if (exceptionalEvidenceFeatures.size() <= nonExceptionalEvidenceFeatures.size()) {
				return new SemanticUnitStub("this is #unusual for " + exceptionalEvidenceFeaturesString + " , #because " + (oneE ? "it is" : "these are") + " not #normally " + (oneE ? "a" : "") + " #strong indicator" + (oneE ? "" : "s") + " ; for " + (oneN ? nonExceptionalEvidenceFeaturesString + " ," : "#the_rest") + " it is #expected");
			}
			else {
				return new SemanticUnitStub("this is #expected for " + nonExceptionalEvidenceFeaturesString + " , #because " + (oneN ? "it is" : "these are") + " #normally " + (oneN ? "a" : "") + " #strong indicator" + (oneN ? "" : "s") + " for predictions of this class" + (oneN ? "" : "s") + " ; for " + (oneE ? exceptionalEvidenceFeaturesString + " ," : "#the_rest") + " it is #unusual");
			}
		}
		else {
			return new SemanticUnitStub("this is #expected , as " + (one ? "it is" : "these are") + " #normally important for predictions of this class");
		}
	}

	private List<Feature> getFeatures(Collection<Feature> keyFeatures, FeatureSelector featureSelector) {
		List<Feature> features = new ArrayList<Feature>();
		for (Feature feature : keyFeatures) {
			if (featureSelector.select(feature)) {
				features.add(feature);
			}
		}
		return features;
	}

	private String makeFeaturesString(Collection<Feature> features) {
		String string = "";

		for (Feature feature : features) {
			string += _featureEmphasizer.decorate(feature.getName()) + ", ";
		}

		if (string.isEmpty()) return null;

		string = string.substring(0, string.length() - 2);
		if (string.lastIndexOf(", ") != -1) {
			string = string.substring(0, string.lastIndexOf(", ")) + " and " + string.substring(string.lastIndexOf(", ") + 2);
		}

		return string;
	}


	private static interface FeatureSelector {
		
		boolean select(Feature feature);
		
	}

}
