package edu.columbia.preju.app;

import java.io.FileReader;
import java.util.Collection;

import ob.core.Feature;
import edu.columbia.preju.generator.HtcJustificationGenerator;
import edu.columbia.preju.generator.JustificationGenerator;
import edu.columbia.preju.generator.JustificationNarrative;
import edu.columbia.preju.generator.JustificationNarrativeImpl;
import edu.columbia.preju.keyFeatureSelection.KeyFeatureSelection;
import edu.columbia.preju.keyFeatureSelection.NonNegligibleKeyFeatureSelection;
import edu.columbia.preju.prediction.Prediction;
import edu.columbia.preju.roleAssignment.RoleAssignment;
import edu.columbia.preju.roleAssignment.TableRoleAssignment;
import edu.columbia.preju.roleAssignment.discretizer.ImportancePercentageDiscretizer;
import edu.columbia.preju.weka.WekaPredictionFactory;
import weka.classifiers.functions.Logistic;
import weka.core.Instance;
import weka.core.Instances;

/**
 * @author Or
 *
 * This class serves as an example for using PreJu as an API. An HTML justification will be produced into the 
 * output directory (by default, this is $home/working).
 */
public class WekaSample {

	public static void main(String[] args) throws Exception {
		String outputDir = "working";
		
		if (args.length == 1) {
			outputDir = args[0];
		}
		if (args.length > 1) {
			System.out.println("only one arg (output directory) allowed");
			System.exit(0);
		}

		
		// Start here. First, let's get a prediction from a Weka classifier
		Prediction prediction = getPrediction();
		
		// The prediction can be exported to an XML file. This file can be used with the command line tool (GenerateJustification), 
		// or loaded into an object using the XmlPredictionFactory
		prediction.exportToXml("working/prediction.xml");

		// The classifier of the prediction can also be exported as a subject domain for the OrbGen package. This must be done once for 
		// each classifier before its predictions can be justified
		prediction.exportSubjectDomainPackage();

		// Second, choose a narrative role assignment method. We'll use the usual table role assignment, with
		// simple threshold discretizers of value 1.0 for both effect and importance (i.e., the measure will be 
		// considered "high positive" if it is higher than 1.0, "high negative" if it is less than -1.0, and "low"
		// otherwise
		RoleAssignment roleAssignment = new TableRoleAssignment(new ImportancePercentageDiscretizer(0.5, prediction));
//		RoleAssignment roleAssignment = new TableRoleAssignment(new FixedNumberDiscretizer(3, prediction));
//		RoleAssignment roleAssignment = new TableRoleAssignment(new ThresholdDiscretizer(1.0));
		
		// Third, choose a key feature selection implementation. Here we'll select all features that 
		// are not Negligible (that is, features with a narrative role other than Negligible)
		KeyFeatureSelection keyFeatureSelection = new NonNegligibleKeyFeatureSelection(); //new HighestEffectKeyFeatureSelection(2);

		// Use the prediction, role assignment and key feature selection to produce a justification narrative
		JustificationNarrative narrative = new JustificationNarrativeImpl(prediction, keyFeatureSelection, roleAssignment); 

		// Show which features were selected as key features and their roles
		System.out.println("all features: " + toString(prediction.getFeatures()));
		System.out.println();
		System.out.println("selected key features: " + toString(narrative.getKeyFeatures()));
		System.out.println();
		System.out.println("assigned narrative roles: ");
		for (Feature feature : narrative.getKeyFeatures()) {
			System.out.println(" - " + feature.getName() + ": " + roleAssignment.getNarrativeRole(feature, prediction));
		}
		System.out.println();

		// Create a generator. This particular one (HTC: Html Text and Chart) generates an html page 
		// called justification.html into the supplied outputDir. The second parameter, the subjectDomain,
		// is the name of the prediction domain (used by the generation package)
		JustificationGenerator generator = new HtcJustificationGenerator(outputDir);

		// Finally, use the generator to generate a justification from the justification narrative
		generator.generate(narrative);
	}

	private static String toString(Collection<Feature> features) {
		String string = "";
		
		for (Feature feature : features) {
			string += ", " + feature.getName();
		}
		
		return "{" + string.substring(2) + "}";
	}

	private static Prediction getPrediction() throws Exception {
		// This is the classifier (multi-class logistic regression)
		Logistic logistic = new Logistic();

		// Read the instances from file and set the class attribute
		Instances instances = new Instances(new FileReader("samples/vote.arff"));
		instances.setClass(instances.attribute("class"));

		// Arbitrary separation to train and test
		Instances train = instances.trainCV(10, 1);
		Instances test = instances.testCV(10, 1);

		// Train the classifier
		logistic.buildClassifier(train);

		// Choose an instance to predict and justify
		Instance instance = test.instance(2);

		// Get a Prediction object
		return WekaPredictionFactory.getLogisticPrediction("vote", logistic, train, instance);
	}

}
