/*
 * unxskt.c: unix socket
 *
 * Hanhua Feng
 * $Id: unxskt.c,v 1.12 2003/05/21 14:59:39 hanhua Exp $
 */
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "htutil.h"
#include "htconfig.h"
#include "htzipd.h"

void enable_conn_input( htman *man, taskentry *tsk )
{
    if ( tsk->sys.sock >= man->sys.nfds )
        man->sys.nfds = tsk->sys.sock + 1;
    FD_SET( tsk->sys.sock, &(man->sys.rfds) );
}

void disable_conn_input( htman *man, taskentry *tsk )
{
    FD_CLR( tsk->sys.sock, &(man->sys.rfds) );
}

void enable_conn_output( htman *man, taskentry *tsk )
{
    if ( tsk->sys.sock >= man->sys.nfds )
        man->sys.nfds = tsk->sys.sock + 1;
    FD_SET( tsk->sys.sock, &(man->sys.wfds) );
}

void disable_conn_output( htman *man, taskentry *tsk )
{
    FD_CLR( tsk->sys.sock, &(man->sys.wfds) );
}

void end_conn( htman *man, taskentry *tsk )
{
    FD_CLR( tsk->sys.sock, &(man->sys.rfds) );
    FD_CLR( tsk->sys.sock, &(man->sys.wfds) );
    close( tsk->sys.sock );
}


static int init_conn( htman *man, int sock )
{
    taskentry *tsk;

    /* setting this socket to non-blockable */
    if ( -1 == fcntl( sock, F_SETFL, O_NONBLOCK ) )
    {
        perror( "fcntl()" );
        close( sock );
    }

    tsk = task_new( int_to_hex_str( sock ) );
    if ( !tsk )
        return -1;

    tsk->sys.sock = sock;

    /* MSS: maximum segment size */
    tsk->sys.mss = 1436;

    start_task( man, tsk );
    return 0;
}

static void write_conn( htman *man, taskentry *tsk )
{
    int r;
    int sock = tsk->sys.sock;

    if ( 0 == tsk->out_len )
        return;

    r = send( sock, tsk->out_buf, tsk->out_len, 0 );

    if ( r < 0 && EAGAIN == errno )
        return;

    if ( r <= 0 )
    {
        end_task( man, tsk );
        return;
    }

    gettimeofday( &(tsk->sys.in_time), 0 );

    if ( r < tsk->out_len )
    {
        tsk->out_len -= r;
        memmove( tsk->out_buf, tsk->out_buf + r, tsk->out_len );
    }
    else
        tsk->out_len = 0;

    /* reading next block in the task */
    if ( tsk->state == TASK_OUTPUT_END || tsk->state == TASK_NOFILE
         || tsk->out_len < tsk->sys.mss )
        exec_task( man, tsk ); /* tsk may have been deleted */
}

static void read_conn( htman *man, taskentry *tsk )
{
    int sock = tsk->sys.sock;

    int r = recv( sock, tsk->in_buf + tsk->in_head,
                  tsk->in_buf_size - tsk->in_head, 0 );
    if ( r < 0 && EAGAIN == errno )
        return;

    if ( r <= 0 )
    {
        end_task( man, tsk );
        return;
    }

    tsk->in_head += r;
    gettimeofday( &(tsk->sys.in_time), 0 );

    if ( REQ_OK == exec_task( man, tsk ) )
    {
        /* send the buffer as soon as possible. */
        write_conn( man, tsk );
    }
}

static int httpd_init( int port )
{
    int lis_sock;
    struct sockaddr_in sktin;

    /* openning a TCP socket for listening */
    if ( ( lis_sock = socket( AF_INET, SOCK_STREAM, 0 ) ) < 0 )
    {
        perror( "socket()" );
        return -1;
    }

    /* binding a TCP socket to the www port ) */
    sktin.sin_family = AF_INET;
    sktin.sin_port = htons( port );
    sktin.sin_addr.s_addr = INADDR_ANY;
    if ( bind( lis_sock, (struct sockaddr *)&sktin, sizeof(sktin) ) < 0 )
    {
        perror( "bind()" );
        return -1;
    }

    /* setting the listenning socket to nonblockable */
    if ( fcntl( lis_sock, F_SETFL, O_NONBLOCK ) );

    /* starting listenning */
    if ( listen( lis_sock, 5 ) < 0 )
    {
        perror( "listen()" );
        return -1;
    }

    return lis_sock;
}

static int httpd( htman *man, int lis_sock )
{
    fd_set rfds, wfds;
    int new_sock;
    int slen, rv;
    struct sockaddr_in sktin;
    struct timeval tv;
    taskentry *tsk, *next_tsk;

    /* initiating socket-related fields in 'man' */
    man->sys.nfds = lis_sock+1;
    FD_ZERO( &(man->sys.rfds) );
    FD_ZERO( &(man->sys.wfds) );

    FD_SET( lis_sock, &(man->sys.rfds) );

    for ( ;; )
    {
        memcpy( &rfds, &(man->sys.rfds), sizeof(rfds) );
        memcpy( &wfds, &(man->sys.wfds), sizeof(wfds) );
        tv.tv_sec = 0;
        tv.tv_usec = 10000;
        rv = select( man->sys.nfds, &rfds, &wfds, NULL, &tv );

        if ( rv < 0 )
        {
            if ( errno == EINTR )
                continue;  /* non blocked signals are ignored */
            perror( "select()" );
            return -1;
        }

        if ( rv )
        {
            if ( FD_ISSET( lis_sock, &rfds ) )
            {
                slen = sizeof(sktin);
                new_sock = accept( lis_sock,
                                   (struct sockaddr *)&sktin, &slen );
                if ( -1 == new_sock )
                    continue;

                if ( -1 == init_conn( man, new_sock ) )
                {
                    fprintf( stderr, "conn_init(): failed.\n" );
                    close( new_sock );
                }
                else
                {
                    /* it seems there is nothing more to do */
                }
            }

            /* check read tasks */
            tsk = man->input;
            if ( tsk )
            {
                do
                {
                    next_tsk = tsk->next;
                    if ( next_tsk == man->input )
                        next_tsk = 0;

                    if ( FD_ISSET( tsk->sys.sock, &rfds ) )
                        read_conn( man, tsk );
                    tsk = next_tsk;
                }
                while ( tsk );
            }

            /* check write tasks */
            tsk = man->output;
            if ( tsk )
            {
                do
                {
                    next_tsk = tsk->next;
                    if ( next_tsk == man->output )
                        next_tsk = 0;

                    if ( FD_ISSET( tsk->sys.sock, &wfds ) )
                        write_conn( man, tsk );
                    tsk = next_tsk;
                }
                while ( tsk );
            }
        }
        else
        {
            resource_collect( man, 0 );
        }

        // do simple scheduling
    }

    return 0;
}

htman my_htman;

int main( int argc, char *argv[] )
{
    extern char *optarg;
    extern int optind;

    int c;
    int sock;
    int port = 80;
    int daemon = 0;
    pid_t pid;
    char *dir = 0;

    /* reading commmand line arguments */
    while ( -1 != ( c = getopt( argc, argv, "d:p:x" ) ) )
    {
        switch ( c )
        {
        case 'p':
            port = atoi( optarg );
            break;
        case 'd':
            dir = optarg;
            break;
        case 'x':
	    daemon = 1;
        default:
            break;
        }
    }

    if ( !dir || optind < argc )
    {
        printf( "Usage: %s [-p <port>] [-v] -d <root> \n", argv[0] );
        return 0;
    }

    htman_init( &my_htman, dir );

    /* initiating configuration */
    CONFIG_INIT();
    validate_config( CONFIG, my_htman.work_dir );

    /* openning the www port */
    sock = httpd_init( port );
    if ( -1 == sock )
    {
        fprintf( stderr, "%s: port %d is not available.\n", argv[0], port );
        return -1;
    }

    /* becoming a daemon */
    if ( daemon )
    {
	/* forking itself. Then the parent exists. */
	if ( ( pid = fork() ) )
	{
	    printf( "PID: %d\n", (int)pid );
	    return 0;
	}
	/*
	 * quitting from its process group, i.e. to remove relationship
	 * with the terminal.
	 */
	setpgid( 0, 0 );

	/* block some of the signals (don't block SIGTERM) */
	signal( SIGHUP, SIG_IGN );  /* hangup of term / death of the parent */
	signal( SIGINT, SIG_IGN );  /* interrupt from keyboard */
	signal( SIGQUIT, SIG_IGN ); /* quit from keyboard */
	signal( SIGALRM, SIG_IGN ); /* signal from alarm */
	signal( SIGURG, SIG_IGN );  /* urgent condition on socket */
	signal( SIGTSTP, SIG_IGN ); /* stop typed at TTY */
	signal( SIGTTIN, SIG_IGN ); /* tty input for background process */
	signal( SIGTTOU, SIG_IGN ); /* tty output for background process */
    }
    signal( SIGPIPE, SIG_IGN ); /* broken pipe */

    httpd( &my_htman, sock );

    htman_cleanup( &my_htman );

    /* cleaning up configuration */
    CONFIG_CLEANUP();

    return 0;
}
