/*
 * task.c: process a http request
 * 
 * Hanhua Feng
 * $Id: task.c,v 1.13 2003/05/21 14:59:39 hanhua Exp $
 */
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "htutil.h"
#include "zip.h"
#include "htzipd.h"

#define IN_BUF_MAXSIZE 16384
#define OUT_BUF_MAXSIZE 8192

taskentry *task_new( const char *name )
{
    taskentry *tsk;
    int len = strlen( name );

    tsk = htmalloc( TASK_STORAGE, sizeof(taskentry)
        + ( IN_BUF_MAXSIZE + OUT_BUF_MAXSIZE + len + 1 ) * sizeof(char) );
    if ( !tsk )
        return 0;

    tsk->name = (void *)(tsk+1);
    strcpy( tsk->name, name );

    tsk->state = TASK_NONE;
    tsk->count = 0;

    tsk->in_buf_size = IN_BUF_MAXSIZE;
    tsk->in_head = 0;
    tsk->in_tail = 0;
    tsk->in_buf = tsk->name + len + 1;

    tsk->out_buf_size = OUT_BUF_MAXSIZE;
    tsk->out_len = 0;
    tsk->out_buf = tsk->in_buf + IN_BUF_MAXSIZE;

    tsk->file = 0;
    tsk->offset = 0;

    tsk->persist_conn = 0;

    task_reset_request( tsk );

    return tsk;
}

void task_delete( taskentry *tsk )
{
    htfree( TASK_STORAGE, tsk );
}

void task_reset_request( taskentry *tsk )
{
    tsk->request_method = HTTP_NONE;
    tsk->request_uri = 0;
    tsk->request_port = 80;
    tsk->request_host = 0;
    tsk->http_version = 0;
    tsk->query = 0;
    tsk->accept = 0;
    tsk->server_host = 0;
    tsk->user_agent = 0;
    tsk->range_num = 0;

    if ( tsk->in_head > tsk->in_tail )
    {
        tsk->in_head -= tsk->in_tail;
        memmove( tsk->in_buf, tsk->in_buf + tsk->in_tail, tsk->in_head );
        tsk->in_tail = 0;
    }
    else
    {
        tsk->in_head = tsk->in_tail = 0;
    }
}

static void get_http_ranges( taskentry *tsk, const char *p )
{
    int i;

    while ( ' ' == *p ) p++;
    if ( 0 != strncmp( p, "bytes=", 6 ) )
        return;

    p += 6;

    for ( i=0; i<sizeof(tsk->range)/sizeof(tsk->range[0]); i++ )
    {
        while( ' ' == *p ) p++;

        if ( '-' == *p )
        {
            p++;
            tsk->range[i].last = (size_t)strtol( p, (char **)&p, 10 );
            tsk->range[i].first = (size_t)-1;
        }
        else if ( isascii(*p) && isdigit((int)*p) )
        {
            tsk->range[i].first = (size_t)strtoul( p, (char **)&p, 10 );
            while ( ' ' == *p ) p++;
            if ( '-' == *p )
            {
                p++;
                while ( ' ' == *p ) p++;
                if ( isascii(*p) && isdigit((int)(*p)) )
                    tsk->range[i].last = 1 +
                        (size_t) strtoul( p, (char **)&p, 10 );
                else
                    tsk->range[i].last = (size_t)-1;
            }
            else
                tsk->range[i].last = (size_t)(-1);
        }
        else
            break;

        while( ' ' == *p ) p++;
        if ( ',' == *p )
            p++;
        else
        {
            i++;
            break;
        }
    }

    tsk->range_num = i;
}

int task_analyze_in_buf( taskentry *tsk )
{
    size_t nbuf;
    char *p, *pline;

    assert( tsk->in_head >= tsk->in_tail );
    for ( ; tsk->in_head > tsk->in_tail; )
    {
        /* scanning for either CR or else LF */
        pline = tsk->in_buf + tsk->in_tail;
        nbuf = tsk->in_head - tsk->in_tail;
        p = memchr( pline, '\r', nbuf );
        if ( !p )
            p = memchr( pline, '\n', nbuf );
        if ( !p )
            return REQ_CONT; /* this line is not ended, return */

        /* check the CRLF pair */
        if ( '\r' == *p
                && p < tsk->in_buf + tsk->in_head - 1
                && '\n' == p[1] )
            *p++ = '\0';;
        *p++ = '\0';

        /* in_tail moves to the next line */
        tsk->in_tail = p - tsk->in_buf;

        /* checking whether the extracted line is a empty line */
        if ( '\0' == *pline )
        {
            if ( tsk->request_method == HTTP_NONE )
                continue; /* ignore empty lines before METHOD token */
            /* or else an empty line indicates the end of message */
            return REQ_END;
        }

        /* bypassing front whitespaces */
        while ( *pline == ' ' ) pline++;

        if ( tsk->request_method == HTTP_NONE )
        {   /* It should be a request line */
            /* the first token */
            p = strchr( pline, ' ' );
            if ( !p )
                return REQ_ERR; /* indicating a request error */
            *p++ = '\0';

            /* checking the http method */
            tsk->request_method = get_http_req_method( pline );
            if ( tsk->request_method == HTTP_UNKNOWN_METHOD )
                return REQ_ERR;

            pline = p;
            while ( *pline == ' ' ) pline++;

            /* extracting request URI and HTTP version*/
            p = strchr( pline, ' ' );
            tsk->request_uri = pline;
            tsk->http_version = 0;
            if ( !p )
                return REQ_END;
            *p++ = '\0';
            str_to_upper( p );
            p = strstr( p, "HTTP/" );
            if ( !p )
                return REQ_END;
            p += 5;
            tsk->http_version = (int) strtol( p, &p, 10 ) << 8;
            p++;
            tsk->http_version += (int) strtol( p, &p, 10 );

            if ( tsk->http_version < 0x100 )
                return REQ_END; /* HTTP version 0.9 or lower */
            if ( tsk->http_version >= 0x101 )
                tsk->persist_conn = 1;
        }
        else
        {   /* other request header fields */
            enum HttpRequest hr;

            if ( *pline == ':' )
                return REQ_ERR;
            p = strchr( pline, ':' );
            if ( !p )
                return REQ_ERR;

            *p++ = '\0';
            p = str_trim( p );

            hr = get_http_header( pline );
            if ( hr == HTTP_UNKNOWN_HEADER )
                continue;

            switch ( hr )
            {
            case HTTP_ACCEPT:
                tsk->accept = p;
                break;
            case HTTP_HOST:
                tsk->server_host = p;
                break;
            case HTTP_USER_AGENT:
                tsk->user_agent = p;
                break;
            case HTTP_RANGE:
                get_http_ranges( tsk, p );
                break;
            case HTTP_CONNECTION:
                str_to_lower( p );
                if ( 0 == strcmp( "close", p ) )
                    tsk->persist_conn = 0;
                else if ( 0 == strcmp( "keep-alive", p ) )
                    tsk->persist_conn = 2;
                break;
            default:
                break;
            }
        }
    }

    return REQ_CONT;
}

static size_t task_gen_content_range_str( taskentry *tsk, char *buf )
{
    int f_size = tsk->file->size;
    sprintf( buf, "Content-Range: bytes %u-%u/%u",
        tsk->range[0].first, tsk->range[0].last-1, f_size );
    return tsk->range[0].last - tsk->range[0].first;
}

static void task_print_head( taskentry *tsk, htman *man )
{
    size_t cont_len;
    assert( 0 == tsk->out_len );

    tsk->out_len = sprintf(
        tsk->out_buf,
        "HTTP/1.1 %s\r\nServer: %s/%s\r\nDate: %s\r\n",
        tsk->range_num ? "206 Partial Content" : "200 OK",
        APPLICATION_NAME, VERSION,
        get_http_date_str( time(0) ) );

    tsk->out_len += sprintf(
        tsk->out_buf + tsk->out_len,
        "Last-Modified: %s\r\nAge: 1800\r\n",
        get_http_date_str(
            tsk->file->time < man->time_start ? man->time_start
            : tsk->file->time ) );

    tsk->out_len += sprintf( tsk->out_buf + tsk->out_len,
        "Content-Type: %s\r\n", get_file_media_type( tsk->file->name ) );

    if ( tsk->range_num )
    {
        cont_len = task_gen_content_range_str(
                tsk, tsk->out_buf + tsk->out_len );
        tsk->out_len += strlen( tsk->out_buf + tsk->out_len );
        tsk->out_len += sprintf( tsk->out_buf + tsk->out_len,
            "\r\nContent-Length: %u\r\n\r\n", cont_len );
    }
    else
    {
        tsk->out_len += sprintf( tsk->out_buf + tsk->out_len,
            "Content-Length: %u\r\n\r\n", tsk->file->size );
    }
}

int task_output_init( taskentry *tsk, htman *man, int *err_code )
{
    size_t f_size = tsk->file->size;

    if ( tsk->range_num > 1 )
    {
        *err_code = HTTP416;
        return REQ_ERR;
    }
    if ( tsk->range_num == 1 )
    {
        if ( tsk->range[0].first == (size_t) -1 )
        {
            tsk->range[0].first = f_size - tsk->range[0].last;
            tsk->range[0].last = f_size;
        }

        if ( tsk->range[0].last > f_size )
            tsk->range[0].last = f_size;
        if ( tsk->range[0].first >= tsk->range[0].last )
        {
            *err_code = HTTP416;
            return REQ_ERR;
        }
    }

    task_print_head( tsk, man );
    tsk->offset = 0;
    return REQ_OK;
}


static int task_load_file_segment( taskentry *tsk, size_t off_end )
{
    fileentry *fe = tsk->file;
    size_t len, b_len;

    if ( !fe )
        return REQ_ERR;
    assert( tsk->out_buf_size >= tsk->out_len );
    b_len = tsk->out_buf_size - tsk->out_len;
    if ( 0 == b_len )
        return REQ_CONT;
    if ( off_end <= tsk->offset )
        return REQ_END;
    len = off_end - tsk->offset;
    if ( len < b_len )
        b_len = len;

    if ( REQ_ERR == fileentry_load( tsk->file, tsk->out_buf + tsk->out_len,
                   b_len, tsk->offset ) )
        return REQ_ERR;

    tsk->offset += b_len;
    tsk->out_len += b_len;

    if ( len > b_len )
        return REQ_CONT;
    return REQ_END;
}

int task_load_file( taskentry *tsk )
{
    int i, rv;
    size_t off_end = tsk->file->size;

    if ( tsk->offset >= tsk->file->size )
        return REQ_END;

    if ( tsk->range_num > 0 )
    {
        /* ranges have been sorted and confined in the file size */
        for ( i=0; i<tsk->range_num; i++ )
        {
            if ( tsk->offset < tsk->range[i].first )
                tsk->offset = tsk->range[0].first;
            if ( tsk->offset < tsk->range[i].last )
            {
                off_end = tsk->range[i].last;
                break;
            }
        }
        if ( i == tsk->range_num )
            return REQ_END;
    }

    rv = task_load_file_segment( tsk, off_end );
    if ( REQ_ERR == rv )
        return REQ_ERR;
    if ( REQ_END == rv && ( tsk->range_num == 0 || i == tsk->range_num - 1 ) )
        return REQ_END;
    return REQ_CONT;
}
