/*
 * mime.c: handling MIME types
 *
 * Hanhua Feng
 * $Id: mime.c,v 1.7 2003/05/21 14:59:39 hanhua Exp $
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "htutil.h"
#include "rc_template.h"
#include "htzipd.h"

#define DEF_MEDIA_TYPE "application/octet-stream"
#define DIR_MEDIA_TYPE "text/html"
#define NOEXT_MEDIA_TYPE "text/plain"
#define ICON_MEDIA_TYPE "image/gif"

#define DEF_MEDIA_ICON "file"
#define DIR_MEDIA_ICON "folder"
#define ZIP_MEDIA_ICON "zip"
#define NOEXT_MEDIA_ICON "file"

typedef struct tag_extentry
{
    char *name;
    char *type;
    char *icon;

    struct tag_extentry *hash_next;
} extentry;

typedef struct {
    int num;
    int tablesize;
    extentry **table;
} exthash;

typedef struct
{
    char *type;
    char *exts;
    char *icon;
} mediaext;

mediaext def_mediaext[] = {
    { "application/msword", "doc", "doc" },
    { "application/pdf", "pdf", "doc" },
    { "application/octet-stream", "bin exe class dll dat", "bin" },
    { "application/postscript", "eps ps", "doc" },
    { "application/vnd.ms-excel", "xls", "doc" },
    { "application/vnd.ms-powerpoint", "ppt", "doc" },
    { "application/x-dvi", "dvi", "doc" },
    { "application/x-gzip", "gz tgz", "bin" },
    { "application/x-tex", "latex tex ltx", "doc" },
    { "application/x-tar", "tar" },
    { "application/x-tcl", "tcl", "text" },
    { "application/x-sh", "sh", "text" },
    { "application/x-csh", "csh", "text" },
    { "application/x-perl", "pl", "text" },
    { "application/x-texinfo", "texinfo texi", "doc" },
    { "application/zip", "zip", "zip" },
    { "audio/basic", "au snd", "audio" },
    { "audio/midi", "mid midi", "audio" },
    { "audio/mpeg", "mpga mp2 mp3", "audio" },
    { "audio/x-realaudio", "ram rm ra", "audio" },
    { "audio/x-wav", "wav", "audio" },
    { "image/bmp", "bmp", "image" },
    { "image/gif", "gif", "image" },
    { "image/jpeg", "jpg jpe jpeg", "image" },
    { "image/png", "png", "image" },
    { "image/tiff", "tiff tif", "image" },
    { "image/x-cmu-raster", "ras", "image" },
    { "image/x-portable-anymap", "pnm", "image" },
    { "image/x-portable-bitmap", "pbm", "image" },
    { "image/x-portable-graymap", "pgm", "image" },
    { "image/x-portable-pixmap", "ppm", "image" },
    { "image/x-rgb", "rgb", "image" },
    { "image/x-xbitmap", "xbm", "image" },
    { "image/x-xpixmap", "xpm", "image" },
    { "model/vrml", "wrl vrml" },
    { "text/css", "css" },
    { "text/html", "html htm", "html" },
    { "text/plain", "asc txt", "text" },
    { "text/plain", "c h cpp cxx cc bas pas f asm pl js pro java bat mak", "text" },
    { "text/richtext", "rtx", "doc" },
    { "text/rtf", "rtf", "doc" },
    { "text/sgml", "sgml sgm" },
    { "text/xml", "xml xsl" },
    { "video/mpeg", "mpeg mpg mpe", "video" },
    { "video/quicktime", "qt mov", "video" },
    { "video/x-msvideo", "avi", "video" }
};

/* iconentry icons[] = { { 0, 0, 0 } }; */

iconentry icons[] = {
    { "?folder.gif", RC_GET(gif_folder), RC_SIZE(gif_folder) },
    { "?zip.gif", RC_GET(gif_zip), RC_SIZE(gif_zip) },
    { "?file.gif", RC_GET(gif_file), RC_SIZE(gif_file) },
    { "?html.gif", RC_GET(gif_html), RC_SIZE(gif_html) },
    { "?bin.gif", RC_GET(gif_bin), RC_SIZE(gif_bin) },
    { "?doc.gif", RC_GET(gif_doc), RC_SIZE(gif_doc) },
    { "?text.gif", RC_GET(gif_text), RC_SIZE(gif_text) },
    { "?image.gif", RC_GET(gif_image), RC_SIZE(gif_image) },
    { "?audio.gif", RC_GET(gif_audio), RC_SIZE(gif_audio) },
    { "?video.gif", RC_GET(gif_video), RC_SIZE(gif_video) },
    { "?zipfolder.gif", RC_GET(gif_zipfolder), RC_SIZE(gif_zipfolder) },
    { "?zipzip.gif", RC_GET(gif_zip), RC_SIZE(gif_zip) },
    { "?zipfile.gif", RC_GET(gif_zipfile), RC_SIZE(gif_zipfile) },
    { "?zipbin.gif", RC_GET(gif_zipbin), RC_SIZE(gif_zipbin) },
    { "?zipdoc.gif", RC_GET(gif_zipdoc), RC_SIZE(gif_zipdoc) },
    { "?ziptext.gif", RC_GET(gif_ziptext), RC_SIZE(gif_ziptext) },
    { "?ziphtml.gif", RC_GET(gif_ziphtml), RC_SIZE(gif_ziphtml) },
    { "?zipimage.gif", RC_GET(gif_zipimage), RC_SIZE(gif_zipimage) },
    { "?zipaudio.gif", RC_GET(gif_zipaudio), RC_SIZE(gif_zipaudio) },
    { "?zipvideo.gif", RC_GET(gif_zipvideo), RC_SIZE(gif_zipvideo) },
    { "?viewzip.gif", RC_GET(gif_viewzip), RC_SIZE(gif_viewzip) },
    { 0, 0, 0 }
};

static exthash mime;

void mime_init( void )
{
    int i;
    size_t len;
    char *p, *pt;
    extentry *mt;

    HASH_INIT( mime, 100 );
    for ( i=0; i<sizeof(def_mediaext)/sizeof(def_mediaext[0]); i++ )
    {
        for ( p = def_mediaext[i].exts; *p; p = pt )
        {
            pt = strchr( p, ' ' );
            if ( pt )
                len = pt - p;
            else
            {
                len = strlen( p );
                pt = p + len;
            }

            if ( 0 == len )
            {
                pt++;
                continue;
            }

            mt = malloc( sizeof(extentry) + len + 1 );
            if ( !mt )
                return;
            mt->type = def_mediaext[i].type;
            mt->icon = def_mediaext[i].icon;
            mt->name = (void *)(mt+1);
            memcpy( mt->name, p, len );
            mt->name[len] = '\0';

            HASH_ADD( mime, mt->name, mt );
#ifdef DEBUG_MIME
            fprintf( stderr, "%s(%d): .%s %s\n", __FILE__, __LINE__,
                     mt->name, mt->type );
#endif
            while ( ' ' == *pt ) pt++;
        }
    }
}

void mime_cleanup( void )
{
    HASH_CLEANUP( mime );
}

const char *get_file_media_type( const char *str )
{
    char buf[8];
    char * p;
    extentry *mt;
    size_t len = strlen( str );

    if ( ( len > 0 && '/' == str[len-1] ) || strchr( str, '*' ) )
        return DIR_MEDIA_TYPE;

    p = strrchr( str, '.' );

    if ( !p )
        return NOEXT_MEDIA_TYPE;

    /* assumming NULL is the smallest pointer */
    if ( p < strrchr( str, '/' ) )
        return NOEXT_MEDIA_TYPE;

    p++;
    if ( strlen(p) >= sizeof(buf)-1 )
        return DEF_MEDIA_TYPE;

    strcpy( buf, p );

    str_to_lower( buf );

    HASH_FIND( mime, buf, mt );
    if ( !mt )
        return DEF_MEDIA_TYPE;

    return mt->type;
}

const char *get_file_media_icon( const char *str, enum FileType ft )
{
    /* str does not necessarily include the full path */
    char buf[8];
    char * p;
    extentry *mt;
    size_t len = strlen( str );

    switch ( ft )
    {
    case FT_DIR:
    case FT_ZIPDIR:
        return DIR_MEDIA_ICON;
    case FT_ZIP:
        return ZIP_MEDIA_ICON;
    default:
        break;
    }

    if ( ( len > 0 && '/' == str[len-1] ) || strchr( str, '*' ) )
        return DIR_MEDIA_ICON;

    p = strrchr( str, '.' );

    if ( !p )
        return NOEXT_MEDIA_ICON;

    /* assumming NULL is the smallest pointer */
    if ( p < strrchr( str, '/' ) )
        return NOEXT_MEDIA_ICON;

    p++;
    if ( strlen(p) >= sizeof(buf)-1 )
        return DEF_MEDIA_ICON;

    strcpy( buf, p );

    str_to_lower( buf );

    HASH_FIND( mime, buf, mt );
    if ( !mt || !mt->icon )
        return DEF_MEDIA_ICON;

    return mt->icon;
}
