#ifndef _HTZIPD_H
#define _HTZIPD_H

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "htutil.h"
#include "zip.h"

#define VERSION "0.1.6"
#define APPLICATION_NAME "LiteZipper"

#define REQ_CONT 0
#define REQ_OK   1
#define REQ_END  2
#define REQ_ERR  (-1)

#define FILE_STORAGE ((void *)(storage_counter+3))
#define TASK_STORAGE ((void *)(storage_counter+4))


/* This structure describes files(including zipfiles) openned */
typedef struct tag_fileentry {
    int count;
    enum FileType ft;
    char *name;  /* DIR and ZIP have "/" tails */
    union {
        void *p;
        FILE *fp;
        memblock *mem;
        const char *cp;
        zipfile *zip;
        fileinfo *fi;
        zipentry *ze;
        zipsession *zs;
    } p;

    time_t time;
    size_t size;

    struct tag_fileentry *base;
    struct tag_fileentry *prev, *next, *hash_next;
} fileentry;

#define MAX_PATH_NUM 16
#define MAX_HTTP_RANGE 8
#define MAX_HTTP_IN_BUF 8192
#define MAX_HTTP_OUT_BUF 8192

#define HTTP_HEADER_ID 0x100

enum HttpRequest {
    HTTP_NONE, HTTP_OPTION, HTTP_GET, HTTP_HEAD, HTTP_POST, HTTP_PUT,
    HTTP_DELETE, HTTP_TRACE, HTTP_CONNECT, HTTP_UNKNOWN_METHOD,

    HTTP_ACCEPT = HTTP_HEADER_ID, HTTP_ACCEPT_CHARSET, HTTP_ACCEPT_ENCODING,
    HTTP_ACCEPT_LANGUAGE, HTTP_AUTHORIZATION, HTTP_EXPECT, HTTP_FROM,
    HTTP_HOST, HTTP_IF_MATCH, HTTP_IF_MODIFIED_SINCE, HTTP_IF_NONE_MATCH,
    HTTP_IF_RANGE, HTTP_IF_UNMODIFIED_SINCE, HTTP_MAX_FORWARDS,
    HTTP_PROXY_AUTHORIZATION,HTTP_RANGE, HTTP_REFERER, HTTP_TE,
    HTTP_USER_AGENT,

    HTTP_CACHE_CONTROL, HTTP_CONNECTION, HTTP_DATE, HTTP_PRAGMA,
    HTTP_TRAILER, HTTP_TRANSFER_ENCODING, HTTP_UPGRADE, HTTP_VIA,
    HTTP_WARNING,

    HTTP_UNKNOWN_HEADER,
};

enum HttpStatus {
    HTTP100 = 1, HTTP101, HTTP200, HTTP201, HTTP202, HTTP203, HTTP204,
    HTTP205, HTTP206, HTTP300, HTTP301, HTTP302, HTTP303, HTTP304, HTTP305,
    HTTP307, HTTP400, HTTP401, HTTP402, HTTP403, HTTP404, HTTP405, HTTP406,
    HTTP407, HTTP408, HTTP409, HTTP410, HTTP411, HTTP412, HTTP413, HTTP414,
    HTTP415, HTTP416, HTTP417, HTTP500, HTTP501, HTTP502, HTTP503, HTTP504,
    HTTP505, HTTP_STATUS_NUM
};

enum TaskState {
    TASK_NONE, TASK_INPUT, TASK_OUTPUT, TASK_OUTPUT_END, TASK_NOFILE
};

typedef struct tag_taskentry {
    /* task state */
    enum TaskState state;
    int count;
    time_t timeout;

    /* buffers */
    size_t in_buf_size;
    size_t in_head;   /* The position to save incoming char from net */
    size_t in_tail;   /* The next unprocessed char */
    char *in_buf;

    size_t out_buf_size;
    size_t out_len;   /* How many chars are valid in out_buf */
    char *out_buf;

    /* the file which needs to be accessed, overwrite file->ft */
    enum FileType ft;
    fileentry *file;
    size_t offset;          /* current offset */

    /* connection parameters */
    enum HttpRequest request_method;
    char *request_uri;
    char *request_host;
    char *new_location;
    int request_port;
    char *query;
    int http_version;
    char *accept;
    char *server_host;
    char *user_agent;
    int range_num;
    struct { size_t first, last; } range[MAX_HTTP_RANGE];

    int persist_conn;
/*
    size_t content_length;
    const char *content_type;
    const char *location;
*/
    /* system-related variables */
    conninfo sys;

    /* storage management */
    char *name;
    struct tag_taskentry *prev, *next, *hash_next, *temp_next;
} taskentry;


typedef struct {
    int num;
    int tablesize;
    taskentry **table;
} taskhash;

typedef struct {
    int num;
    int tablesize;
    fileentry **table;
} filehash;


typedef struct {
    taskhash tasks;
    filehash files;

    taskentry *input;
    taskentry *output;

    const char *work_dir;

    time_t time_start;
    netinfo sys;
} htman;

/* functions in htmlgen.c */

size_t html_from_template( char *buf, size_t size, const char *templ,
    size_t (*_func_inc)(char *,size_t,const char *,void *), void *param );
/* html_from_template generates html(or other text as well ) from a
 * template html text. The template contains strings started with '$'
 * which will be replaced with corresponding texts in configuration, and
 * string started with '%' which will be replaced by calling function
 * _func_inc. Output string stores in 'buf' and will not exceed 'size'
 * bytes including the terminal ASCIIZ. 'templ' is the template text.
 * param will be passed to _func_inc as the forth parameter. The first
 * three parameters of _func_inc are the buffer, its size, and the
 * string following %. A string can be enclosed in (), where () is not
 * part of string itself, e.g. $(Abc) %(def). When the string is
 * not enclosed in (), we include all alphabets, numbers and '_'s as
 * many as possible for '$' while we include only one char for '%'.
 */

memblock *html_generate( const char *templ );
/* html_generate calls html_from template to generate HTML pages
 * without providing _func_inc. It returns an memblock structure.
 */

memblock *html_dir_list( const char *dir, const char *uri );
memblock *html_dir_list_sorted( fileinfo *fi_dir, const char *uri,
                                const char *so );
/* html_dir_list and html_dir_list_sorted generate HTML dir list
 * according to the content of dir 'dir'. The title is shown as
 * 'uri' rather than 'dir' itself. It returns a memblock structure
 * if successed, or NULL.
 * html_dir_list_sorted reads content from an fileino array and
 * uses an index sorted with an order specified by string 'so'.
 */

memblock *html_zipdir_list( zipentry *first, const char *uri );
memblock *html_zipdir_list_sorted( zipentry *first, const char *uri,
                                   const char *so );
/* html_zip_dir_list generates HTML dir list according to the content
 * of a dir of a zipfile. The first file of this dir is discribed by
 * a 'zipentry' first, which leads a doubly linked list of the whole
 * content.
 * html_dir_list_sorted uses an index sorted with an order specified
 * by string 'so'.
 */

/* functions in htproc.c */

enum HttpRequest get_http_req_method( char *str );
/* get_http_req_method returns the request id of the method string */

enum HttpRequest get_http_header( char *str );
/* get_http_req_header returns the header id of the header string */

char *uri_hex_decode( char *str );
/* URI_hex_decode removes %HexHex in str and returns overwrited str
 * It returns str in any case.
 */

char *uri_hex_encode( const char *str );
/* URI_hex_encode changes char which can not exist in URL to %HexHex
 * in str and returns a static internal buffer.
 */

char *split_uri( char *uri, char **host, int *port, char **query );
/* split_uri extracts path and args from URI. It returns the uri path
 * overwrited on original uri and passes back host name, port and query.
 */

char *filt_path( char *uri_path );
/* filt_path corrects the uri path. It possibly removes invalid chars.
 * it would remove double slashs but keep trailing '/' unchanged. It
 * also try to remove /./ and /../ to avoid security problems. We assumed
 * caller has decoded %HexHex codes in URI. uri_path will be overwrited.
 */

char *realize_path( const char *uri_path, const char **virtual_table );
/* realize_path checks virtual dirs table and returns a static buffer where
 * the absolute path of the host is stored. vitual_table stores a string
 * table. The odd string is a virtual dir and the even string before it
 * is its actual dir.  A virtual dir with which a uri_path starts
 * should be replace with its corresponding actual dir. virtual_table
 * ended with a NULL string and the last single actual dir is the default
 * directory for "/". virtual_table would be checked sequentially.
 * It returns NULL if failed.
 */

char *get_index_page( const char *abs_path );
/* get_index_page returns a path of an index file (typically 'index.html')
 * to an '/'-trailed path. It will return NULL if abs_path is not
 * '/'-trailed.
 */


const char *check_path_security( const char *abs_path,
                                 const char **prefix_table );
/* check_path_security checks whether abs_path begins with one of the
 * string in prefix_table. prefix_table ended with a NULL string.
 * Ordinarily prefix_table should be specified by command-line, rather
 * than configured through HTTP. If prefix_table is a NULL, the
 * security check will be bypassed. It returns NULL if failed.
 */

/* functions in file.c */

void fileentry_delete_object( enum FileType ft, void *p );
/* fileentry_delete_object deletes the object managed by fileentry */

void fileentry_delete( fileentry *fe );
/* fileetnry_delete removes a fileentry as well as its memory storage. */

void fileentry_inc( fileentry *fe );
void fileentry_dec( fileentry *fe );
/* These two functions increases and decreases the reference count
 * of a fileentry
 */

fileentry *fileentry_static_init( htman *man, char *name,
                                  const char *buf, size_t size );
/* fileentry_static_init loads a static resource to a fileentry for its
 * accessibility via file hash.
 */

fileentry *fileentry_memblock_init( htman *man, char *name, memblock *m );
/* fileentry_memblock_init create a fileenry for a memblock. */


fileentry *fileentry_init( htman *man, const char *path, const char *uri,
                           const char *query, int *app_slash );
/* fileentry_init checks the type of path: a dir, a file or in a zipfile
 * It returns openned fileentry structure. app_slash points to the address
 * where a flag will be returned to show whether a slash '/' should be
 * appended to uri.
 * for case-insesitive OSs, the return string contains only lower cases ???
 * This function will check information on the disk.
 * It returns NULL if the path is not assessible.
 */

int fileentry_load( fileentry *fe, char *buf, size_t size, size_t offset );
/* fileentry_load read 'size' bytes data from the source specified by 'fe'
 * at the offset 'offset'.
 */

/* functions in task.c */

taskentry *task_new( const char *name );
/* task_new allocates a task structure and initiates it. */

void task_delete( taskentry *tsk );

void task_reset_request( taskentry *tsk );
/* task_reset_request resets all request parameters. It is used before
 * analyzing the next request if a persistent connection is allowed in
 * HTTP 1.1
 */

int task_analyze_in_buf( taskentry *tsk );
/* analyze_in_buf reads and analyzes input buffer as much as possible.
 * it returns REQ_END if the request is ended, REQ_ERR if there is an
 * error in the request, or REQ_CONT if the request has not finished.
 */

void task_gen_status( taskentry *tsk, enum HttpStatus status );

int task_load_file( taskentry *tsk );
/* task_load_file returns REQ_END, REQ_CONT, or REQ_ERR */

int task_output_init( taskentry *tsk, htman *man, int *err_code );
/* task_output_init initiate response buffer by adding header information
 * in it. So it may use the information from structure htman. It will return
 * REQ_OK or REQ_ERR;
 */

/* functions in error.c */

void error_response_init( htman *man );
/* error_response_init function loads all error message from resources */

size_t generate_error_response( /* htman *man, */ taskentry *tsk,
                                char *buf, size_t buf_size, int err_code,
                                int f_entity );
/* generate_error_response generate error response to the output buffer */

/* functions in mime.c */

typedef struct { char *name; const char *buf; size_t size; } iconentry;

extern iconentry icons[];

void mime_init( void );
/* mime_init initiates MIME structures and hash tables */

void mime_cleanup( void );
/* mime_cleanup cleans up all MIME structures */

const char *get_file_media_type( const char *str );
/* get_file_media_type returns the MIME type string of a file ext 'str' */

const char *get_file_media_icon( const char *str, enum FileType ft );
/* get_file_media_icon returns the icon of file 'str' with type 'ft' */

/* functions in httpd.c */

void start_task( htman *man, taskentry *tsk );
/* start_task added a new connection into 'man'. tsk has been allocated
 * by task_new. start_task will add tsk into the hash table. It also changes
 * the state of tsk to input mode.
 */

void end_task( htman *man, taskentry *tsk );
/* end_task will remove 'tsk' from 'man'. It will also delete tsk by
 * calling task_delete.
 */

int exec_task( htman *man, taskentry *tsk );
/* exec_task does a lot of things. When tsk is with input state, exec_task
 * analyzes the input request. If the request finishes, exec_task will
 * change the state to output mode and read the first output data from
 * files. When tsk is in output state, it tries to fill the output buffer.
 * If the file reaches an end, it will change the state to OUTPUT_END
 * state.
 * In input mode, This routine returns REQ_OK to tell the caller that the
 * input request has been processed and the out_buf should be sent as soon
 * as possible, or REQ_CONT to send in the next turn.  In output mode, it
 * returns REQ_OK to indicate that data has been sent but the connection
 * remains to accept next request, and REQ_CONT to indicate that there are
 * more data are waiting to be sent. In both mode, it returns REQ_END to
 * indicate the task has been ended and the task has been deleted.
 */

void reset_task( htman *man, taskentry *tsk );
/* reset_task changes the task state from OUTPUT_END state(or something
 * else) to input state. it removes task from output list and adds task
 * to input state.
 */

void htman_init( htman *man, const char *work_dir );
/* htman_init initiates structures and variables in the htman structure */

void htman_cleanup( htman *man );
/* htman_cleanup release all resources */

void resource_collect( htman *man, int level );
/* collect unused / cached file/dir handle and zipfile structure */

/* system dependent functions in either unxskt.c or winskt.c */

void enable_conn_input( htman *man, taskentry *tsk );
void disable_conn_input( htman *man, taskentry *tsk );
void enable_conn_output( htman *man, taskentry *tsk );
void disable_conn_output( htman *man, taskentry *tsk );
/* These functions enable/disable socket input/output polling */

void end_conn( htman *man, taskentry *tsk );
/* close the connection */

#endif /* _HTZIPD_H */
