#ifndef _HTUTIL_H
#define _HTUTIL_H

#include <stdlib.h>
#include <time.h>

#if defined(WIN32)

#include <windows.h>
#include <winsock.h>

#elif defined(UNIX)

#include <unistd.h>
#include <dirent.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>

#endif

/* hooked functions */

void *htmalloc( void *param, size_t size );
void *htcalloc( void *param, size_t n, size_t size );
void htfree( void *param, void *buffer );
void *htrealloc( void *param, void *buffer, size_t size );
FILE *htfopen( const char*fn, const char *mode );
void htfclose( FILE *fp );

#define STORAGE_NUM 5
extern int storage_counter[];

/* generally available macros and functions (not system specific) */

#ifndef min
#define min(a,b) ((a)>(b)?(b):(a))
#endif

char *str_to_upper( char *str );
char *str_to_lower( char *str );
char *str_trim( char *str );
/* all these functions returns original buffer */

char *int_to_hex_str( unsigned int n );

typedef struct {
    char *ptr;
    size_t size;
} memblock;

memblock *memblock_new( size_t size );
memblock *memblock_realloc( memblock *m, size_t size );
void memblock_delete( memblock *m );

#define MEMBLOCK_STORAGE ((void *)(storage_counter+1))

int hash_func( const char *str, int n );

/* Doubly linked list operations (cycled) */
#define DLINKED_ADD(p,head)                                             \
    do {                                                                \
        if ( head ) {                                                   \
            (p)->next = (head);                                         \
            (p)->prev = head->prev;                                     \
            (p)->prev->next = (p)->next->prev = (p);                    \
        } else (head) = (p)->prev = (p)->next = (p);                    \
    } while(0)

#define DLINKED_REMOVE(p,head)                                          \
    do {                                                                \
        if ( (p)->prev != (p) ) {                                       \
            (p)->next->prev = (p)->prev; (p)->prev->next = (p)->next;   \
            if ( (p) == (head) ) (head) = p->next;                      \
        } else (head) = 0;                                              \
    } while(0)

#define DLINKED_ITERATE(head,p,func)                                    \
    do {                                                                \
        p = head;                                                       \
        if ( p )                                                        \
            do { func; p=p->next; } while ( p!=head );                  \
    } while(0)


/* Hash table operations */

#define HASH_INIT(obj,n)                                                \
    do {                                                                \
        (obj).tablesize = n; (obj).num = 0;                             \
        (obj).table = calloc( (obj).tablesize, sizeof(*(obj).table) );  \
        if ( !(obj).table ) abort();                                    \
    } while(0)

#define HASH_CLEANUP(obj)                                               \
    do {                                                                \
        void *p; int i;                                                 \
        for ( i = (obj).tablesize-1; i>=0; i-- )                        \
            for ( p = (obj).table[i]; p; p = (obj).table[i] )           \
                { (obj).table[i]=(obj).table[i]->hash_next; free(p); }  \
        (obj).num = 0; free( (obj).table ); (obj).table = 0;            \
    } while(0)

#define HASH_CLEANUP_WITH(obj,p,func)                                   \
    do {                                                                \
        int i;                                                          \
        for ( i = (obj).tablesize-1; i>=0; i-- )                        \
            for ( p = (obj).table[i]; p; p = (obj).table[i] )           \
                { (obj).table[i]=(obj).table[i]->hash_next; func; }     \
        (obj).num = 0;  free( (obj).table ); (obj).table = 0;           \
    } while(0)

#define HASH_FIND(obj,nm,p)                                             \
    do {                                                                \
        for ( p = (obj).table[hash_func(nm,(obj).tablesize)];           \
            p; p = p->hash_next )                                       \
            if ( 0 == strcmp(p->name,(nm)) ) break;                     \
    } while(0)

#define HASH_ITERATE(obj,p,func)                                        \
    do {                                                                \
        int i;                                                          \
        for ( i = (obj).tablesize-1; i>=0; i-- )                        \
            for ( p = (obj).table[i]; p; p = p->hash_next )             \
                { func; }                                               \
    } while(0)

#define HASH_FIND_NEXT(obj,nm,p)                                        \
    do {                                                                \
        for ( ; p; p = p->hash_next )                                   \
            if ( 0 == strcmp(p->name,(nm)) ) break;                     \
    } while(0)

#define HASH_ADD(obj,nm,p)                                              \
    do {                                                                \
        int i = hash_func((nm),(obj).tablesize);                        \
        (p)->hash_next = (obj).table[i]; (obj).table[i] = (p);          \
        (obj).num++;                                                    \
    } while(0)

#define HASH_REMOVE(obj,nm,p)                                           \
    do {                                                                \
        int i=hash_func((nm),(obj).tablesize); void *q = 0, *pt = p;    \
        for ( p=(obj).table[i]; p; q=p, p=p->hash_next )                \
            if ( pt == p ) break;                                       \
        if ( p ) {                                                      \
            void *qt = p->hash_next;                                    \
            if ( q ){ p=q; p->hash_next = qt; }                         \
            else (obj).table[i] = qt;                                   \
            (obj).num--;                                                \
        }                                                               \
        p = pt;                                                         \
    } while(0)


/* CPU based macros and functions */

#if defined(SOLARIS) || defined(IRIX)
#define get_intel16(x) ((((x)>>8)&0xff)|(((x)<<8)&0xff00))
#define get_intel32(x) ((((x)>>24)&0xff)|(((x)>>8)&0xff00)              \
                       |(((x)<<8)&0xff0000)|(((x)<<24)&0xff000000))
#elif defined(LINUX) || defined(WIN32)
#define get_intel16(x) (x)
#define get_intel32(x) (x)
#else
#error You must defined the CPU Endian!
#endif


/* ==== Operating system based functions ==== */
#if defined(WIN32)   /* Windows systems */

#ifndef MAX_PATH
#define MAX_PATH 1024
#endif

typedef FILETIME systime;

#define SYSTIME_IN_MSEC(t) ((*(LONGLONG *)(t))/10000)
#define SYSTIME_IN_SEC(t) ((*(LONGLONG *)(t))/10000000)

typedef struct {
    int count;
    HANDLE h;
    WIN32_FIND_DATA d;
} directory;

typedef struct {
    SOCKET sock;
    int mss;
    int flags;
} conninfo;

#define CONN_IO_CHANGE 1
#define CONN_INPUT     2
#define CONN_OUTPUT    4

typedef struct {
    HWND hwnd;
    SOCKET sock;
} netinfo;

#elif defined(UNIX)  /* Unix systems */

#ifndef MAX_PATH
#define MAX_PATH 1024
#endif

typedef struct timeval systime;

#define SYSTIME_IN_MSEC(t) ((t).sec*1000+(t).msec)
#define SYSTIME_IN_SEC(t) ((t).sec)

typedef struct {
    DIR *dir;
    size_t path_len;
    char path[MAX_PATH];
} directory;

typedef struct {
    int sock;      /* connection socket */
    int mss;       /* MSS: maximum segment size */
    struct timeval in_time;
    struct timeval out_time;
} conninfo;

typedef struct {
    int nfds;      /* number of opened files */
    fd_set rfds;
    fd_set wfds;
} netinfo;

#endif

/* time operations */

#define SYSTIME_DIFF_IN_MSEC(t1,t2) ((int)TIME_IN_MSEC(t2)-(int)TIME_IN_MSEC(t1))
#define SYSTIME_DIFF_IN_SEC(t1,t2) ((int)TIME_IN_SEC(t2)-(int)TIME_IN_SEC(t1))

time_t systime_to_time( systime t );

time_t get_dos_time( unsigned short dos_date, unsigned short dos_time );

char *get_http_date_str( time_t t );

/* File system operations */

enum FileType {
    FT_NONE, FT_OTHER, FT_FILE, FT_ZIP, FT_DIR, FT_MEMORY, FT_STATIC,
    FT_ZIPFILE, FT_ZIPDIR
};

typedef struct {
    enum FileType ft;
    time_t ctime;
    time_t mtime;
    time_t atime;
    size_t size;
    const char *name;
} fileinfo;

enum FileType get_file_info( const char *file, fileinfo *fi );
/* get_file_info return the information of file 'file' to struct fileinfo.
 * fi->name on return will point to the filename in the buffer of 'file'.
 * The path will not be included in fi->name.
 */

int dir_open( directory *di, const char *name );
/* dir_open opens directory 'name' to directory structure *di
 * It returns -1 if there is an error occurred.
 */

void dir_close( directory *di );
/* dir_close closes the directory openned by dir_open */

int dir_read( directory *di, fileinfo *fi );
/* dir_read reads the next item in the directory *di. It returns
 * information to structure fileinfo *fi. Note that fi->name will
 * point to a static buffer which could be overwrite on next calling
 * or become invalid after dir_close is called. fi->name includes
 * only the filename, the path is excluded. It returns -1 if there
 * is an error occurred.
 */

#define MAX_DIR_NAMEBUF (2048*16)
/* The maximum total bytes of name buffer in a directory. */

#define MAX_DIR_ENTRYNUM 2048
/* Maximum number of entries in a directory */

#define FILEINFO_STORAGE ((void *)(storage_counter+1))

fileinfo *dir_load( const char *dir_name );
/* dir_load reads all content of a dir into memory. It saves these
 * information to an array of struct fileinfo, which was allocated
 * by itself. This allocated buffer contains not only the array
 * terminated by a FT_NONE fileinfo structure, but also the names
 * of files (entries in a dirs), excluding the path name.
 */

#endif /* _HTUTIL_H */

