/*
 * htutil.c: basic supporting library
 *
 * Hanhua Feng
 * $Id: htutil.c,v 1.11 2003/05/21 14:59:39 hanhua Exp $
 */
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "htutil.h"

int storage_counter[STORAGE_NUM] = { 0, 0, 0, 0, 0 };

void *htmalloc( void *param, size_t size )
{
    ++(*(int *)param);
    return malloc( size );
}

void *htcalloc( void *param, size_t n, size_t size )
{
    ++(*(int *)param);
    return calloc( n, size );
}

void htfree( void *param, void *buffer )
{
    --(*(int *)param);
    free( buffer );
}

void *htrealloc( void *param, void *buffer, size_t size )
{
    (*(int *)param) = (*(int *)param);
    return realloc( buffer, size );
}

FILE *htfopen( const char*fn, const char *mode )
{
    FILE *fp = fopen( fn, mode );
    if ( fp ) ++storage_counter[0];
    return fp;
}

void htfclose( FILE *fp )
{
    --storage_counter[0];
    fclose( fp );
}


char *str_to_upper( char *str )
{
    int i;
    for ( i=0; str[i]; i++ )
        str[i] = toupper(str[i]);
    return str;
}

char *str_to_lower( char *str )
{
    int i;
    for ( i=0; str[i]; i++ )
        str[i] = tolower(str[i]);
    return str;
}

char *str_trim( char *str )
{
    int i;
    char *p = str;
    while( isascii(*p) && isspace(*p) )
        p++;
    for ( i = strlen(p); i>0 && isascii(p[i-1]) && isspace(p[i-1]); i-- )
        ;
    p[i] = '\0';
    return p;
}

#define int_to_HEX(i) ((i)>=10?(i)+('A'-10):(i)+'0')

char *int_to_hex_str( unsigned int n )
{
    int i;
    static char buf[10];

    for ( i=0; i<8; i++ )
    {
        buf[i] = int_to_HEX( n >> 28 );
        n <<= 4;
    }
    buf[i] = '\0';
    return buf;
}


memblock *memblock_new( size_t size )
{
    memblock *m = htmalloc( MEMBLOCK_STORAGE, sizeof(memblock) + size );
    m->ptr = (void *)(m+1);
    m->size = size;
    return m;
}

memblock *memblock_realloc( memblock *m, size_t size )
{
    m->size = size;
    return htrealloc( MEMBLOCK_STORAGE, m, size + sizeof(memblock) );
}

void memblock_delete( memblock *m )
{
    htfree( MEMBLOCK_STORAGE, m );
}


int hash_func( const char *str, int n )
{
    unsigned int s, x;
    const char *p;

    s = 0;
    for ( p = str; *p; p++ )
    {
        s = ( s<<4 ) + (*p);
        if ( ( x = s & 0xf0000000 ) )
        {
            s ^= x>>24;
            s ^= x;
        }
    }

    return s % (unsigned int)n;
}

/*
 * Windows should have a function CoDosDateTimeToFileTime
 * time_t get_dos_time( unsigned short dos_date, unsigned short dos_time )
 * {
 *     systime t;
 *     if ( FALSE == CoDosDateTimeToFileTime( dos_date, dos_time, &t ) )
 *          return 0;
 *     return systime_to_time( t );
 * }
 *
*/

time_t get_dos_time( unsigned short dos_date, unsigned short dos_time )
{
    struct tm t;
    t.tm_year = ( ( dos_date >> 9 ) & 0x7f ) + 80;
    t.tm_mon = ( ( dos_date >> 5 ) & 0xf ) - 1;
    t.tm_mday = ( dos_date & 0x1f ) - 1;
    t.tm_hour = ( dos_time >> 11 ) & 0x1f;
    t.tm_min = ( dos_time >> 5 ) & 0x3f;
    t.tm_sec = ( dos_time & 0x1f ) << 1;

    return mktime( &t );
}

char *get_http_date_str( time_t t )
{
    static char *week_str[7] =
        { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
    static char *mon_str[12] = {
        "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug",
        "Sep", "Oct", "Nov", "Dec" };
    static char buf[32];
    struct tm *gm_tm = gmtime( &t );

    if ( gm_tm )
    {
        sprintf( buf, "%.3s, %02d %.3s %04d %02d:%02d:%02d GMT",
            week_str[gm_tm->tm_wday], gm_tm->tm_mday,
            mon_str[gm_tm->tm_mon], gm_tm->tm_year + 1900,
            gm_tm->tm_hour, gm_tm->tm_min, gm_tm->tm_sec );
    }
    else
        buf[0] = '\0';
    return buf;
}


#if defined(WIN32)

time_t systime_to_time( systime syst )
{
    /* FILETIME is based on 1601 and time_t 1970
     * FILETIME is in 100ns and time_t 1s
     */
    unsigned long hi, lo, l, q, r;

    /* substract difference between 1601 and 1970 */
    hi = syst.dwHighDateTime;
    l = lo = syst.dwLowDateTime;
    hi += 0xfe624e9e - 0x7d;
    lo += 0xe4fbaeb0 - 0xba821800;  /* Found 15 hours difference */
    if ( lo < l )
        hi++;

    /* 5^7 is a little greater than 2^16 */
    q = hi / 78125;  /* q must be smaller than 2^16 */

    /* q should be smaller than 2^7, or the division will overflow */
    if ( q >= ( 1 << 7 ) )
        return (time_t) -1;

    r = hi % 78125;  /* r must be smaller than 2^17 */
    hi = ( r << 15 ) | ( lo >> 17 );
    q <<= 15;
    q += hi / 78125;
    r = hi % 78125;
    hi = ( r << 15 ) | ( ( lo >> 2 ) & ( ( 1 << 15 ) - 1 ) );
    l = hi / 78125;

    /* It should be q<<=15 and q += l. After that there are 2 bits remained
     * in the dividend. Since we will divide 2^7 again, that means shift
     * towards right by 7. Equivalently we discard remain two bits and
     * shift towards right by 5. So we do q<<10 and q += l >> 5. Finally
     * we check the highest bit shifted out to decide how to round.
     */
    q <<= 10;
    q += l >> 5;
    if ( l & ( 1 << 4 ) )
        q++;
    return q;
}

#elif defined(UNIX)

time_t systime_to_time( systime syst )
{
    return syst.tv_sec;
}

#endif

#if defined(WIN32)

enum FileType get_file_info( const char *file, fileinfo *fi )
{
/*
    const char *p;
    struct _stat st;
    if ( _stat( file, &st ) )
        return FT_NONE;

    fi->mtime = st.st_mtime;
    fi->atime = st.st_atime;
    fi->ctime = st.st_ctime;
    fi->size = st.st_size;

    fi->name = strrchr( file, '/' );
    p = strrchr( file, '\\' );

    if ( p && fi->name && fi->name - p > 0 )
        p = fi->name;

    if ( p )
        fi->name = p + 1;
    else
        fi->name = file;

    if ( st.st_mode & _S_IFDIR )
        fi->ft = FT_DIR;
    else if ( st.st_mode & _S_IFREG )
        fi->ft = FT_FILE;
    return fi->ft;
*/

    const char *p;
    HANDLE h;
    WIN32_FIND_DATA wfd;
    DWORD attr = GetFileAttributes( file );

    if ( 0xFFFFFFFF == attr )
        return FT_NONE;

    if ( attr & FILE_ATTRIBUTE_DIRECTORY )
    {
        fi->mtime = fi->atime = fi->ctime = 0;
        fi->ft = FT_DIR;
        fi->size = 0;
    }
    else
    {
        h = FindFirstFile( file, &wfd );
        FindClose( h );
        if ( INVALID_HANDLE_VALUE == h )
            return FT_NONE;

        fi->mtime = systime_to_time( wfd.ftLastWriteTime );
        fi->atime = systime_to_time( wfd.ftLastAccessTime );
        fi->ctime = systime_to_time( wfd.ftCreationTime );

        fi->ft = FT_FILE;
        fi->size = wfd.nFileSizeHigh ? 0xFFFFFFFF : wfd.nFileSizeLow;
    }

    fi->name = strrchr( file, '/' );
    p = strrchr( file, '\\' );

    if ( p && fi->name && fi->name - p > 0 )
        p = fi->name;

    if ( p )
        fi->name = p + 1;
    else
        fi->name = file;

    return fi->ft;
}

#elif defined(UNIX)

enum FileType get_file_info( const char *file, fileinfo *fi )
{
    struct stat st;

    if ( stat( file, &st ) )
        return FT_NONE;

    fi->mtime = st.st_mtime;
    fi->atime = st.st_atime;
    fi->ctime = st.st_ctime;
    fi->size = st.st_size;
    fi->name = strrchr( file, '/' );
    if ( fi->name )
        fi->name++;
    else
        fi->name = file;

    if ( st.st_mode & S_IFDIR )
        fi->ft = FT_DIR;
    else if ( st.st_mode & S_IFREG )
        fi->ft = FT_FILE;
    return fi->ft;
}

#endif


#if defined(WIN32)

int dir_open( directory *di, const char *name )
{
    static char s_name[MAX_PATH];
    int len = strlen( name );
    if ( len > 0 && ( name[len-1] == '/' || name[len-1] == '\\' ) )
        len--;
    if ( len >= MAX_PATH - 3 )
        return -1;
    memcpy( s_name, name, len );
    s_name[len++] = '/';
    s_name[len++] = '*';
    s_name[len] = '\0';
    di->h = FindFirstFile( s_name, &(di->d) );
    if ( di->h != INVALID_HANDLE_VALUE )
        di->count = 0;
    else
        di->count = -1;
    return 0;
}

void dir_close( directory *di )
{
    FindClose( di->h );
}

int dir_read( directory *di, fileinfo *fi )
{
    if ( di->count == -1 )
        return -1;

    for ( ;; )
    {
        if ( di->count > 0 && !FindNextFile( di->h, &(di->d) ) )
        {
            di->count = -1;
            return -1;
        }

        di->count++;

        if ( '.' != di->d.cFileName[0] )
            break;
        if ( '\0' != di->d.cFileName[1]
             && ( '.' != di->d.cFileName[1] || '\0' != di->d.cFileName[2] ) )
            break;
    }

    fi->ctime = systime_to_time( di->d.ftCreationTime );
    fi->mtime = systime_to_time( di->d.ftLastWriteTime );
    fi->atime = systime_to_time( di->d.ftLastAccessTime );
    fi->name = di->d.cFileName;
    if ( di->d.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
    {
        fi->ft = FT_DIR;
        fi->size = 0;
    }
    else
    {
        fi->ft = FT_FILE;
        fi->size = di->d.nFileSizeHigh ? 0xFFFFFFFF : di->d.nFileSizeLow;
    }

    return 0;
}

#elif defined(UNIX)

int dir_open( directory *di, const char *name )
{
    di->path_len = strlen(name);
    if ( di->path_len >= MAX_PATH-1 )
        return -1;
    di->dir = opendir( name );
    if ( !di->dir )
        return -1;
    strcpy( di->path, name );
    if ( di->path_len > 0 && '/' == name[di->path_len-1] )
        di->path_len--;
    return 0;
}

void dir_close( directory *di )
{
    closedir( di->dir );
}

int dir_read( directory *di, fileinfo *fi )
{
    char s_path[MAX_PATH];
    struct dirent *d_ent;
    struct stat f_stat;

    for ( ;; )
    {
        d_ent = readdir( di->dir );
        if ( !d_ent )
            return -1;

        /* ignore '.' and '..' */
        if ( '.' == d_ent->d_name[0]
            && ( ( '.' == d_ent->d_name[1] && '\0' == d_ent->d_name[2] )
                 || '\0' == d_ent->d_name[1] ) )
            continue;

        if ( di->path_len + strlen( d_ent->d_name ) < MAX_PATH - 2 )
            break;
    }

    if ( di->path_len )
        memcpy( s_path, di->path, di->path_len );
    s_path[di->path_len] = '/';

    strcpy( s_path + di->path_len + 1, d_ent->d_name );

    if ( stat( s_path, &f_stat ) >= 0 )
    {
        fi->ctime = f_stat.st_ctime;
        fi->mtime = f_stat.st_mtime;
        fi->atime = f_stat.st_atime;
        if ( f_stat.st_mode & S_IFDIR )
        {
            fi->size = 0;
            fi->ft = FT_DIR;
        }
        else if ( f_stat.st_mode & S_IFREG )
        {
            fi->size = f_stat.st_size;
            fi->ft = FT_FILE;
        }
        else
        {
            fi->size = 0;
            fi->ft = FT_OTHER;
        }
        fi->name = d_ent->d_name;
    }
    else
    {
#ifdef DEBUG
        fprintf( stderr, "%s: not found.\n", s_path );
#endif
        memset( fi, 0, sizeof(*fi) );
        fi->name = d_ent->d_name;
    }
    return 0;
}

#endif

fileinfo *dir_load( const char *dir_name )
{
    int i, n;
    char *p;
    size_t offset, len;
    directory di;
    fileinfo *fi;

    if ( dir_open( &di, dir_name ) < 0 )
        return 0;

    fi = htmalloc( FILEINFO_STORAGE,
        sizeof(fileinfo)*MAX_DIR_ENTRYNUM + MAX_DIR_NAMEBUF );
    if ( !fi )
    {
        dir_close( &di );
        return 0;
    }

    p = (void *)(fi+MAX_DIR_ENTRYNUM);
    offset = 0;

    for ( i=0; i<MAX_DIR_ENTRYNUM-1; i++ )
    {
        if ( dir_read( &di, fi+i ) < 0 )
            break;

        if ( fi[i].ft == FT_FILE )
        {
            const char *p = strstr(fi[i].name, ".zip" );
            if ( !p )
                p = strstr( fi[i].name, ".ZIP" );
            if ( p && '\0' == p[4] )
                fi[i].ft = FT_ZIP;
        }

        len = strlen( fi[i].name );
        if ( len + offset >= MAX_DIR_NAMEBUF - 2 )
            break;
        memcpy( p+offset, fi[i].name, len );
        fi[i].name = p + offset;
        offset += len;
        p[offset++] = '\0';
    }

    fi[i].ft = FT_NONE;

    if ( 0 == i )
        return realloc( fi, sizeof(fileinfo) );
    /* Shrinking a memory block would not fail */

    n = i;

    if ( n < MAX_DIR_ENTRYNUM-1 )
    {
        size_t s_diff = (MAX_DIR_ENTRYNUM-n-1) * sizeof(fileinfo);
        memcpy( (void *)(fi+n+1), p, offset );

        for ( i=0; i<n; i++ )
            fi[i].name -= s_diff;
    }

    return realloc( fi, sizeof(fileinfo)*(n+1) + offset );
}

