/*
 * httpd.c: handling httpd requests
 *
 * Hanhua Feng
 * $Id: httpd.c,v 1.18 2003/05/21 14:59:39 hanhua Exp $
 */
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include "htutil.h"
#include "htconfig.h"
#include "rc_template.h"
#include "zip.h"
#include "htzipd.h"

void htman_init( htman *man, const char *work_dir )
{
    int i;

    /* static char path[MAX_PATH]; */

    /* strncpy( path, dir, MAX_PATH-1 ); */

    HASH_INIT( man->tasks, 10 );
    HASH_INIT( man->files, 200 );

    man->input = 0;
    man->output = 0;

    man->work_dir = work_dir;
    man->time_start = time(0);

    mime_init();
    error_response_init( man );

    for ( i=0; icons[i].name; i++ )
    {
        fileentry_static_init( man, icons[i].name, icons[i].buf, icons[i].size );
    }

    /* field sys has not been initiated. It would be initiated
     * by main program instead of here.
     */
}

void htman_cleanup( htman *man )
{
    fileentry *fe;
    taskentry *tsk;

    mime_cleanup();

    HASH_CLEANUP_WITH( man->tasks, tsk, (task_delete(tsk)) );
    HASH_CLEANUP_WITH( man->files, fe, (fileentry_delete(fe)) );

    man->input = 0;
    man->output = 0;
}



void start_task( htman *man, taskentry *tsk )
{
    tsk->timeout = time(0) + CONFIG_OF(conn_timeout);

    HASH_ADD( man->tasks, tsk->name, tsk );
    tsk->state = TASK_INPUT;
    DLINKED_ADD( tsk, man->input );
    enable_conn_input( man, tsk );
}

void end_task( htman *man, taskentry *tsk )
{
    switch ( tsk->state )
    {
    case TASK_INPUT:
        disable_conn_input( man, tsk );
        DLINKED_REMOVE( tsk, man->input );
        break;
    case TASK_OUTPUT:
    case TASK_OUTPUT_END:
    case TASK_NOFILE:
        disable_conn_output( man, tsk );
        DLINKED_REMOVE( tsk, man->output );
        break;
    default:
        break;
    }

    if ( tsk->file )
        fileentry_dec( tsk->file );

    HASH_REMOVE( man->tasks, tsk->name, tsk );

    end_conn( man, tsk );

    task_delete( tsk );
}

static fileentry *get_static_resource( htman *man, const char *uri,
                                       const char *query )
{
    size_t len;
    char buf[32];
    fileentry *fe;

    if ( strcmp( uri, "/" ) || 0 == query )
        return 0;

    switch ( query[0] )
    {
    case 'r':
        if ( 0 == strncmp( query, "rc=", 3 ) )
        {
            len = strlen( query+3 );
            if ( len > 20 )
                return 0;

            buf[0] = '?';
            strcpy( buf+1, query+3 );
            HASH_FIND( man->files, buf, fe );

            return fe;
        }
        break;

    case 'h':
        if ( 0 == strcmp( query, "help" ) )
        {
            fe = fileentry_memblock_init( man, "??help.html",
                                          html_generate( RC_GET(help) ) );
            return fe;
        }
        break;

    case 'c':
        if ( 0 == strncmp( query, "config", 6 ) )
        {
        }
        break;

    default:
        break;
    }

    return 0;
}

static int _task_output_end( htman *man, taskentry *tsk )
{
    if ( ( tsk->http_version >= 0x101 && tsk->persist_conn )
         || ( tsk->http_version == 0x100 && tsk->persist_conn >= 2 ) )
    {
        tsk->count++;
        reset_task( man, tsk );

        return REQ_OK;
    }


    /* If it's not HTTP1.1 persistent connection, the tsk will be
     * removed from output list by end_task.
     */
    end_task( man, tsk );

    return REQ_END;
}

int exec_task( htman *man, taskentry *tsk )
{
    enum TaskState state;
    int err_code = -1, f_app_slash, uri_len, q_len, rv;
    char *abs_path, *index_path;
    char uri_buf[1024];

    tsk->timeout = time(0) + CONFIG_OF(conn_timeout);

    state = tsk->state;
    switch ( state )
    {
    case TASK_INPUT:
        rv = task_analyze_in_buf( tsk );
        if ( REQ_ERR == rv )
        {
            err_code = HTTP400;  /* bad request */
            break;
        }

        if ( REQ_CONT == rv )
            return REQ_CONT;

        if ( tsk->http_version > 0x101 )
        {
            err_code = HTTP505; /* version not implement */
            break;
        }

        if ( HTTP_POST == tsk->request_method )
        {
            /* POST: not implemented yet */
            err_code = HTTP501; /* not implement */
            break;
        }

        if ( HTTP_GET != tsk->request_method
             && HTTP_HEAD != tsk->request_method )
        {
            err_code = HTTP501; /* not implement */
            break;
        }

        tsk->request_uri = split_uri( tsk->request_uri, &(tsk->request_host),
                                      &(tsk->request_port), &(tsk->query) );
        if ( !tsk->request_uri )
        {
            err_code = HTTP400; /* bad request */
            break;
        }

        if ( tsk->request_host )
        {
            /* PROXY: not implemented yet */
            err_code = HTTP501; /* not implement */
            break;
        }

        uri_len = strlen(tsk->request_uri);
        q_len = tsk->query ? strlen(tsk->query) : 0;
        if ( uri_len + q_len >= sizeof(uri_buf) - 4 )
        {
            err_code = HTTP414; /* request-URI too long */
            break;
        }
        strcpy( uri_buf, tsk->request_uri );

        if ( !filt_path( uri_hex_decode( tsk->request_uri ) ) )
        {
            err_code = HTTP400; /* bad request */
            break;
        }

        /* check whether its a system interpreted URI */
        tsk->file = get_static_resource( man, tsk->request_uri, tsk->query );
        if ( !tsk->file )
        {
            if ( !(abs_path = realize_path( tsk->request_uri,
                                            CONFIG_OF(uri_mapping) ) ) )
            {
                err_code = HTTP404; /* not found */
                break;
            }

            if ( !check_path_security( abs_path, CONFIG_OF(allow_path) ) )
            {
                err_code = HTTP403; /* forbidden */
                break;
            }

            index_path = get_index_page( abs_path );
            if ( index_path )
                tsk->file = fileentry_init( man, index_path, uri_buf,
                                            tsk->query, &f_app_slash );
            else
                tsk->file = 0;

            if ( !tsk->file )
                tsk->file = fileentry_init( man, abs_path, uri_buf,
                                            tsk->query, &f_app_slash );

            if ( f_app_slash )
            {
                err_code = HTTP302; /* found; redirect */
                uri_buf[uri_len++] = '/';
                if ( tsk->query )
                {
                    uri_buf[uri_len++] = '?';
                    strcpy( uri_buf+uri_len, tsk->query );
                }
                else
                    uri_buf[uri_len] = '\0';
                tsk->new_location = uri_buf;
                break;
            }

            if ( !tsk->file )
            {
                err_code = HTTP404; /* not found */
                break;
            }
        }

        /* Other headers will be checked in task_output_init */
        if ( task_output_init( tsk, man, &err_code ) == REQ_ERR )
        {
            /* err_code has been set */
            break;
        }

        if ( HTTP_HEAD == tsk->request_method )
        {
            disable_conn_input( man, tsk );
            DLINKED_REMOVE( tsk, man->input );
            tsk->state = TASK_NOFILE;
            DLINKED_ADD( tsk, man->output );
            enable_conn_output( man, tsk );
            return REQ_OK;
        }

        fileentry_inc( tsk->file );

        disable_conn_input( man, tsk );
        DLINKED_REMOVE( tsk, man->input );
        tsk->state = TASK_OUTPUT;
        DLINKED_ADD( tsk, man->output );
        enable_conn_output( man, tsk );

        /* fall through, load first block */
    case TASK_OUTPUT:
        rv = task_load_file( tsk );

        if ( REQ_ERR == rv )
        {
            if ( tsk->file )
            {
                fileentry_dec( tsk->file );
                tsk->file = 0;
            }
            end_task( man, tsk );
            return REQ_END;
        }

        if ( REQ_END == rv )
        {
            tsk->state = TASK_OUTPUT_END;
        }

        if ( state == TASK_INPUT )
            return REQ_OK;
        return REQ_CONT;

    case TASK_OUTPUT_END:
        if ( 0 == tsk->out_len )
        {
            if ( tsk->file )
            {
                fileentry_dec( tsk->file );

                /* close only file handles when finished */
                if ( ( tsk->file->ft == FT_FILE || tsk->file->ft == FT_ZIPFILE )
                     && 0 == tsk->file->count )
                {
                    HASH_REMOVE( man->files, tsk->file->name, tsk->file );
                    fileentry_delete( tsk->file );
                }

                tsk->file = 0;
            }

            return _task_output_end( man, tsk );
        }
        return REQ_CONT;

    case TASK_NOFILE:  /* Head or configuration */
        if ( 0 == tsk->out_len )
        {
            return _task_output_end( man, tsk );
        }
        return REQ_CONT;

    default:
        fprintf( stderr, "exec_task(): invalid task state.\n" );
        end_task( man, tsk );
        return REQ_END;
    }

    /* remove the reference to a fileentry */
    tsk->file = 0;

    /* The following will be executed only if error occurred */
    if ( err_code >= 0 )
    {
        assert( tsk->state == TASK_INPUT );
        assert( tsk->out_len == 0 );

        tsk->out_len += generate_error_response(
            tsk,
            tsk->out_buf + tsk->out_len,
            tsk->out_buf_size - tsk->out_len,
            err_code, HTTP_GET == tsk->request_method );

        disable_conn_input( man, tsk );
        DLINKED_REMOVE( tsk, man->input );
        tsk->state = TASK_NOFILE;
        DLINKED_ADD( tsk, man->output );
        enable_conn_output( man, tsk );
    }
    else
        abort();
    return REQ_OK;
}

void reset_task( htman *man, taskentry *tsk )
{
    switch ( tsk->state )
    {
    case TASK_OUTPUT:
    case TASK_OUTPUT_END:
    case TASK_NOFILE:
        disable_conn_output( man, tsk );
        DLINKED_REMOVE( tsk, man->output );
        tsk->state = TASK_INPUT;
        DLINKED_ADD( tsk, man->input );
        enable_conn_input( man, tsk );
        break;

    case TASK_INPUT:
        break;
    default:
        break;
    }

    task_reset_request( tsk );
}

void resource_collect( htman *man, int level )
{
    const char *work_dir;

    if ( !man->tasks.table || !man->files.table )
        return;

    if ( man->tasks.num > 0 )
    {
        taskentry *tsk, *first = 0;
        time_t ct = time(0);

        HASH_ITERATE( man->tasks, tsk,
            ((tsk->timeout<ct)?(tsk->temp_next=first, first=tsk):tsk) );
        for ( tsk = first; tsk; tsk = first )
        {
            first = first->temp_next;
            HASH_REMOVE( man->tasks, tsk->name, tsk );
            end_task( man, tsk );
        }
    }

    if ( level >= 3 )
    {
        if ( man->tasks.num <= 0 )
        {
            work_dir = man->work_dir;
            htman_cleanup( man );
            htman_init( man, work_dir );
            return;
        }

        level = 2;
    }

    if ( level )
    {
        fileentry *fe, *first;
        for ( ; level > 0; level-- )
        {
            first = 0;
            HASH_ITERATE( man->files, fe,
                (fe->count==0?(fe->next=first, first=fe):fe) );
            for ( fe = first; fe; fe = first )
            {
                first = first->next;
                assert( fe->ft != FT_STATIC );
                HASH_REMOVE( man->files, fe->name, fe );
                fileentry_delete( fe );
            }
        }
    }
    else
    {
    }
}

