/*
 * htproc.c: processing header information
 *
 * Hanhua Feng
 * $Id: htproc.c,v 1.10 2003/05/21 14:59:39 hanhua Exp $
 */
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "htutil.h"
#include "htconfig.h"
#include "zip.h"
#include "htzipd.h"


enum HttpRequest get_http_req_method( char *str )
{
    str_to_lower( str );

    switch ( *str )
    {
    case 'g':
        if ( 0 == strcmp( str, "get" ) )
            return HTTP_GET;
        break;
    case 'h':
        if ( 0 == strcmp( str, "head" ) )
            return HTTP_PUT;
        break;
    case 'p':
        if ( 0 == strcmp( str, "post" ) )
            return HTTP_POST;
        else if ( 0 == strcmp( str, "put" ) )
            return HTTP_PUT;
        break;
    case 'o':
        if ( 0 == strcmp( str, "option" ) )
            return HTTP_OPTION;
        break;
    case 'd':
        if ( 0 == strcmp( str, "delete" ) )
            return HTTP_DELETE;
        break;
    case 't':
        if ( 0 == strcmp( str, "trace" ) )
            return HTTP_TRACE;
        break;
    case 'c':
        if ( 0 == strcmp( str, "connect" ) )
            return HTTP_CONNECT;
        break;
    default:
        break;
    }
    return HTTP_UNKNOWN_METHOD;
}

enum HttpRequest get_http_header( char *str )
{
    /* HTTP_HEADERS_ID must be alphabetically ascendant */
    static struct { const char *s; enum HttpRequest r; } hdrs[] = {
        { "accept", HTTP_ACCEPT },
        { "accept-charset", HTTP_ACCEPT_CHARSET },
        { "accept-encoding", HTTP_ACCEPT_ENCODING },
        { "accept-language", HTTP_ACCEPT_LANGUAGE },
        { "authorization", HTTP_AUTHORIZATION },
        { "cache-control", HTTP_CACHE_CONTROL }, /* general header */
        { "connection", HTTP_CONNECTION }, /* general header */
        { "date", HTTP_DATE },             /* general header */
        { "expect", HTTP_EXPECT },
        { "from", HTTP_FROM },
        { "host", HTTP_HOST },
        { "if-match", HTTP_IF_MATCH },
        { "if-modified-since", HTTP_IF_MODIFIED_SINCE },
        { "if-none-match", HTTP_IF_NONE_MATCH },
        { "if-range", HTTP_IF_RANGE },
        { "if-unmodified-since", HTTP_IF_UNMODIFIED_SINCE },
        { "max-forwards", HTTP_MAX_FORWARDS },
        { "pragma", HTTP_PRAGMA },         /* general header */
        { "proxy-authorization", HTTP_PROXY_AUTHORIZATION },
        { "range", HTTP_RANGE },
        { "referer", HTTP_REFERER },
        { "te", HTTP_TE },
        { "trailer", HTTP_TRAILER },       /* general header */
        { "transfer-encoding", HTTP_TRANSFER_ENCODING },/* general header */
        { "upgrade", HTTP_UPGRADE },       /* general header */
        { "user-agent", HTTP_USER_AGENT },
        { "via", HTTP_VIA },
        { "warning", HTTP_WARNING }
    };

    int i, n, st = 0, en = sizeof(hdrs)/sizeof(hdrs[0]) - 1;

    str_to_lower( str );

    /* searching in the sorted string array. */
    for ( ; en >= st; )
    {
        i = ( st + en ) / 2;
        n = strcmp( str, hdrs[i].s );
        if ( 0 == n )
            return hdrs[i].r;
        if ( n > 0 )
            st = i + 1;
        else
            en = i - 1;
    }
    return HTTP_UNKNOWN_HEADER;
}

#define hex_to_int(c) \
    ((c)>='a'?((c)-'a'+10):((c)>='A'?((c)-'A'+10):((c)-'0')))

#define int_to_HEX(i) ((i)>=10?(i)+('A'-10):(i)+'0')

char *uri_hex_decode( char *str )
{
    int i, j;
    for ( i=0, j=0; str[i]; i++ )
    {
        if ( '%' == str[i] && str[i+1] && str[i+2] )
        {
            str[j++] = hex_to_int(str[i+1]) * 16 + hex_to_int(str[i+2]);
            i += 2;
        }
        else
            str[j++] = str[i];
    }
    str[j] = '\0';
    return str;
}

char *uri_hex_encode( const char *str )
{
    static char buf[1024];
    int i, j;

    for ( i=0, j=0; str[i] && j<sizeof(buf)-1; i++  )
    {
        if ( !isascii(str[i]) || iscntrl(str[i])
             || strchr( " !@#$%^&*+~`<>?\"\':;", str[i] ) )
        {
            if ( j < sizeof(buf) - 3 )
            {
                buf[j++] = '%';
                buf[j++] = int_to_HEX( ((unsigned char)str[i]) / 16 );
                buf[j++] = int_to_HEX( ((unsigned char)str[i]) % 16 );
            }
        }
        else
            buf[j++] = str[i];
    }
    buf[j] = '\0';
    return buf;
}

char *split_uri( char *uri, char **host, int *port, char **query )
{
    int n;
    char *p, *pt;

    assert( host && port && query );

    if ( '/' == uri[0] )
    {
        /* No host/port, just /the/path?query */
        *host = 0;
        *port = -1;
        p = uri;
    }
    else if ( memcmp( "http://", uri, 7 ) )
        return 0;
    else
    {   /* full URL, start with "http://", 7 chars */
        /* searching for first slash to break host/port and abs path */
        p = strchr( uri+7, '/' );
        if ( !p )
            p = strchr( uri+7, '?' );

        if ( p )
            n = p - (uri+7);
        else
            n = strlen( uri+7 );

        if ( n > 0 )
        {
            /* moving host+port backward in order to add '\0' to the end */
            memmove( uri, uri+7, n );
            uri[n] = '\0';
            *host = uri;

            pt = strchr( uri, ':' );
            if ( pt )
            {
                *pt = '\0';
                *port = atoi( pt+1 );
            }
            else
                *port = 80;
        }
        else
        {
            *host = 0;
            *port = 0;
        }

        if ( !p || *p == '?' )
        {   /* in case no absolute path */
            p = uri + n + 6;
            *p = '/';
        }
    }

    /* splitting path string and query string */
    pt = strchr( p, '?' );
    if ( !pt )
        *query = 0;
    else
    {
        *pt++ = '\0';
        *query = pt;
    }

    pt = strrchr( p, '#' );
    if ( pt )
        *pt = 0;

    return p;
}

char *filt_path( char *uri_path )
{
    int i, j;
    for ( i=0, j=0; uri_path[i]; i++ )
    {
        if ( '/' == uri_path[i] )
        {
            if ( '.' == uri_path[i+1] )
            {
                if ( '/' == uri_path[i+2] || '\0' == uri_path[i+2] )
                {
                    /* "/./" case */
                    i++;  /* just skip "." */
                }
                else if ( '.' == uri_path[i+2]
                          && ( '.' == uri_path[i+3]
                               || '/' == uri_path[i+3]
                               || '\0' == uri_path[i+3] ) )
                {   /* "/../" or "/...xxx/" case */
                    /* uri_path+i moves to the char before the next '/' */
                    for ( i += 2; uri_path[i+1] && '/' != uri_path[i+1]; i++ )
                        ;
                    /* uri_path+j moves back to last '/' or beginng*/
                    for ( j--; j >= 0 && '/' != uri_path[j]; j-- )
                        ;
                }
                else
                    uri_path[j++] = uri_path[i];
            }
            else if ( '/' != uri_path[i+1] )
                uri_path[j++] = uri_path[i];
            /* The current char is skipped '/' if the next is also '/' */
        }
        else
            uri_path[j++] = uri_path[i];
    }

    uri_path[j] = '\0';

    return uri_path;
}

char *realize_path( const char *uri_path, const char **virtual_table )
{
    static char path[MAX_PATH+256];
    /* the path in zipfile should not exceed 256 bytes */
    int i, len = 0, pos = -1;

    for ( i=0; virtual_table[i] && virtual_table[i+1]; i+=2 )
    {
        len = strlen( virtual_table[i] );
        if ( len && 0 == strncmp( virtual_table[i], uri_path, len ) )
        {
            pos = i;
            break;
        }
    }
    if ( pos < 0 )
    {
        /* There is at least one char in each entry of the table */
        if ( virtual_table[i] && virtual_table[i][0] )
            pos = i;
        else
            return 0;
    }

    /* Now pos is the index of virtual table and len is the number
     * of chars to skip from uri_path.
     */
    i = strlen( virtual_table[pos] );
    if ( i + strlen( uri_path+len ) >= MAX_PATH + 254 )
        return 0;

    if ( i>0 && virtual_table[pos][i-1] == '/' )
        i--;

    memcpy( path, virtual_table[pos], i );
    if ( uri_path[len] != '/' )
        path[i++] = '/';
    strcpy( path+i, uri_path+len );

    return path;
}

const char *check_path_security( const char *abs_path,
                                 const char **prefix_table )
{
    int i, n;

    if ( !prefix_table )
        return abs_path;

    for ( i=0; prefix_table[i]; i++ )
    {
        n = strlen( prefix_table[i] );
        if ( 0 == strncmp( prefix_table[i], abs_path, n ) )
            return prefix_table[i];
    }

    return 0;
}

char *get_index_page( const char *abs_path )
{
    static char path[MAX_PATH];
    size_t len = strlen(abs_path);

    if ( len > 0 && abs_path[len-1] != '/' )
        return 0;

    if ( strlen(CONFIG_OF(index_file)) + len >= sizeof(path)-1 )
        return 0;

    memcpy( path, abs_path, len );
    strcpy( path+len, CONFIG_OF(index_file) );

    return path;
}
