/*
 * htmlgen.c: generate HTML files from template files
 *
 * Hanhua Feng
 * $Id: htmlgen.c,v 1.11 2003/05/21 14:59:39 hanhua Exp $
 */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "htutil.h"
#include "zip.h"
#include "htconfig.h"
#include "htzipd.h"

size_t html_from_template( char *buf, size_t size, const char *templ,
    size_t (*_func_inc)(char *,size_t,const char *,void *), void *param )
{
    const char *p, *p1, *p2, *pt;
    char str[64];
    size_t len, off = 0;

    if ( 0 == templ )
    {
        if ( size > 4 )
            strcpy( buf, "N/A" );
        else
            buf[0] = '\0';
        return 0;
    }

    for ( p = templ; ; )
    {
        p1 = strchr( p, '%' );
        p2 = strchr( p, '$' );
        if ( p1 )
        {
            if ( p2 )
                len = ( p2-p1>0 ? p2-p : p1-p );
            else
                len = p1 - p;
        }
        else
        {
            if ( p2 )
                len = p2 - p;
            else
                len = strlen( p );
        }

        if ( len > 0 )
        {
            if ( len + off >= size - 1 )
            {
                memcpy( buf+off, p, size - off - 1 );
                off = size - 1;
                break;
            }

            memcpy( buf+off, p, len );
            off += len;
            p += len;
        }

        if ( !(*p) )
            break;

        if ( '(' == p[1] )
        {
            p += 2;
            p1 = strchr( p, ')' );
            len = p1 - p;
            if ( p1 - p < sizeof(str) - 1 )
            {
                memcpy( str, p, len );
                str[len] = '\0';

                if ( p[-2] == '%' )
                {
                    if ( _func_inc )
                        off += (*_func_inc)( buf+off, size-off, str, param );
                }
                else
                {
                    CONFIG_OF_STR( str, pt );
                    len = strlen( pt );
                    if ( len + off >= size - 1 )
                        break;
                    memcpy( buf+off, pt, len );
                }
            }
            p = p1+1;
        }
        else if ( p[0] == p[1] )
        {
            if ( off < size - 1 )
                buf[off++] = p[0];
            p += 2;
        }
        else if ( '%' == (*p) )
        {
            if ( ! (*p) )
                break;
            str[0] = p[1];
            str[1] = '\0';
            if ( _func_inc )
                off += (*_func_inc)( buf+off, size-off, str, param );
            p += 2;
        }
        else
        {
            p++;
            for ( len=0; len<sizeof(str)-1; len++ )
            {
                if ( !isascii(*p) || ( !isalnum(*p) && '_' != *p ) )
                    break;
                str[len] = *p++;
            }
            str[len] = '\0';

            CONFIG_OF_STR( str, pt );

            if ( pt )
            {
                len = strlen( pt );
                if ( len + off >= size - 1 )
                    break;
                memcpy( buf+off, pt, len );
                off += len;
            }
        }
    }

    buf[off] = '\0';

    return off;
}

memblock *html_generate( const char *templ )
{
    size_t len = strlen( templ );

    memblock *m = memblock_new( len + (len>>2) + 256 );

    len = html_from_template( m->ptr, m->size, templ, 0, 0 );
    return memblock_realloc( m, len );
}

/* Strings showing the html text to display the content of a dir. In these
 * strings, %<c> will be substituted by relative fields, where <c> is a
 * letter or a digit. The exact meaning of this char is:
 *  p: dir name(without path)    P: dir name (with URI path)
 *  R: path with URIs
 *  OD Od: URL sorted by dir
 *  ON On: URL sorted by name
 *  OS Os: URL sorted by size
 *  OT Ot: URL sorted by date
 *  OM Om: URL sorted by MIME type
 *  *: directory text
 * In the directory text content:
 *  b<n>: number of space, used with %n
 *  u: uri link of the file.        U: uri link with slash appended
 *  t: short modification time.     T: long modification time.
 *  d: short modification date.     D: long modification date.
 *  n<n>: the cut name of the file. N: Full name of the file.
 *  s: the short size.              S: Full size of the file.
 *  m[<n>]: the icon
 */

#define PREALLOC_SIZE 65536

#define SORT_BY_DIR_A  'D'
#define SORT_BY_DIR_D  'd'
#define SORT_BY_NAME_A 'N'
#define SORT_BY_NAME_D 'n'
#define SORT_BY_SIZE_A 'S'
#define SORT_BY_SIZE_D 's'
#define SORT_BY_DATE_A 'T'
#define SORT_BY_DATE_D 't'
#define SORT_BY_MIME_A 'M'
#define SORT_BY_MIME_b 'm'

#define HTML_AHREF_PREFIX  "<a href=\""
#define HTML_AHREF_POSTFIX "\">"
#define HTML_AHREF_TERM    "</a>"

struct s_dir_list_item
{
    fileinfo *fi;
    struct tm *ptm;
};

struct s_dir_list
{
    const char *uri;
    fileinfo *(*dir_read)( void *param );
    union {
        directory *pdir;
        fileinfo **fi_index;
        zipentry **ze_index;
        zipentry *ze_head;
    } p1;
    union {
        int at;
        zipentry *ze_at;
    } p2;
};


static char *size_to_short_size( size_t size )
{
    static char buf[8];
    if ( size <= 99999 )
        sprintf( buf, "%5u", size );
    else if ( size < 1024 * 9999 + 512 )
        sprintf( buf, "%4uK", ( size + 512 ) / 1024 );
    else
        sprintf( buf, "%4uM", ( size + 512*1024) / (1024*1024) );
    return buf;
}


static size_t html_dir_list_item_callback( char *buf, size_t size,
                                           const char *id, void *param )
{
    static const char *mon_str[12] = {
        "JAN", "FEB", "MAR", "APR", "MAY", "JUN",
        "JUL", "AUG", "SEP", "OCT", "NOV", "DEC"
    };
    fileinfo *fi = ((struct s_dir_list_item *)param)->fi;
    struct tm *ptm = ((struct s_dir_list_item *)param)->ptm;
    const char *p;
    size_t len, xlen;

    switch ( *id )
    {
    case 'b':
        if ( isascii(id[1]) && isdigit(id[1]) )
        {
            xlen = atoi(id+1);
            if ( xlen < 4 || xlen > 128 )
                xlen = 16;
        }
        else
            xlen = 16;

        len = strlen( fi->name );
        if ( len < xlen )
        {
            len = xlen - len;
            memset( buf, ' ', len );
            return len;
        }
        return 0;

    case 'n':
        if ( isascii(id[1]) && isdigit(id[1]) )
        {
            xlen = atoi(id+1);
            if ( xlen < 4 || xlen > 128 )
                xlen = 16;
        }
        else
            xlen = 16;

        if ( size <= xlen+5 )
            return 0;

        len = strlen( fi->name );
        if ( len >= xlen )
        {
            memcpy( buf, fi->name, xlen-1 );
            buf[xlen-1] = '&';
            buf[xlen] = 'g';
            buf[xlen+1] = 't';
            buf[xlen+2] = ';';
            return xlen+3;
        }
        memcpy( buf, fi->name, len );
        return len;

    case 'N':
        p = fi->name;
        break;

    case 'u':
        p = uri_hex_encode( fi->name );
        break;

    case 'U':
        p = uri_hex_encode( fi->name );
        len = strlen(p);
        if ( len >= size - 2 )
            return 0;
        memcpy( buf, p, len );
        buf[len++] = '/';
        return len;

    case 's':
        p = size_to_short_size( fi->size );
        break;

    case 'S':
        if ( size <= 16 )
            return 0;
        return sprintf( buf, "%d", fi->size );

    case 't':
        if ( size <= 5+4 )
            return 0;
        if ( ptm )
            sprintf( buf, "%02d:%02d", ptm->tm_hour, ptm->tm_min );
        else
            memset( buf, ' ', 5 );
        return 5;

    case 'T':
        if ( size <= 8+4 )
            return 0;
        if ( ptm )
            sprintf( buf, "%02d:%02d:%02d",
                     ptm->tm_hour, ptm->tm_min, ptm->tm_sec );
        else
            memset( buf, ' ', 8 );
        return 8;

    case 'd':
        if ( size <= 8+4 )
            return 0;
        if ( ptm )
            sprintf( buf, "%02d/%02d/%02d", ptm->tm_mon+1,
                     ptm->tm_mday, ptm->tm_year%100 );
        else
            memset( buf, ' ', 8 );
        return 8;

    case 'D':
        if ( size <= 11+4 )
            return 0;
        if ( ptm )
            sprintf( buf, "%.3s %02d %4d", mon_str[ptm->tm_mon],
                     ptm->tm_mday, ptm->tm_year + 1900 );
        else
            memset( buf, ' ', 11 );
        return 11;

    case 'm':
        p = get_file_media_icon( fi->name, fi->ft );
        len = strlen( p );
        if ( len + 16 >= size )
            return 0;
        if ( fi->ft != FT_ZIPFILE && fi->ft != FT_ZIPDIR )
        {
            memcpy( buf, "/?rc=", 5 );
            memcpy( buf+5, p, len );
            strcpy( buf+5+len, ".gif" );
            len += 9;
        }
        else
        {
            memcpy( buf, "/?rc=zip", 8 );
            memcpy( buf+8, p, len );
            strcpy( buf+8+len, ".gif" );
            len += 12;
        }
        return len;

    default:
        return 0;
    }

    len = strlen( p );
    if ( len >= size - 1 )
        return 0;
    memcpy( buf, p, len );
    return len;
}


static size_t html_dir_list_callback( char *buf, size_t size,
                                      const char *id, void *param )
{
    struct s_dir_list_item dli;
    struct s_dir_list *dl = param;
    size_t off, len;
    const char *templ;
    const char *p, *pt;

    switch ( *id )
    {
    case 'p':
        if ( !dl->uri[0] )
            return 0;
        for ( pt = dl->uri; ; pt++ )
        {
            p = pt;
            pt = strchr( pt, '/' );
            if ( !pt || !pt[1] )
                break;
        }
        off = strlen( p );
        if ( p != dl->uri && p[off-1] == '/' )
            off--;
        if ( off > size - 1 )
            break;
        memcpy( buf, p, off );
        break;

    case 'R':
        off = 0;
        for ( p = dl->uri; ; p = pt + 1 )
        {
            while ( '/' == *p )
            {
                if ( off >= size - 2 )
                    return off;
                buf[off++] = '/';
                p++;
            }

            pt = strchr( p, '/' );
            if ( !pt )
            {
                len = strlen( p );
                if ( len + off >= size - 1 )
                    return off;

                memcpy( buf+off, p, len );
                off += len;
                break;
            }

            if ( (pt-dl->uri)*3 + (pt-p) + off + 32 >= size - 1 )
                return off;

            len = strlen(HTML_AHREF_PREFIX);
            memcpy( buf+off, HTML_AHREF_PREFIX, len );
            off += len;

            len = pt - dl->uri;
            memcpy( buf+off, uri_hex_encode(dl->uri), len );
            off += len;
            buf[off++] = '/';

            len = strlen(HTML_AHREF_POSTFIX);
            memcpy( buf+off, HTML_AHREF_POSTFIX, len );
            off += len;

            len = pt - p;
            memcpy( buf+off, p, len );
            off += len;

            len = strlen(HTML_AHREF_TERM);
            memcpy( buf+off, HTML_AHREF_TERM, len );
            off += len;

            buf[off++] = '/';
        }

        break;

    case 'P':
        off = strlen( dl->uri );
        if ( off > size - 1 )
            break;
        memcpy( buf, dl->uri, off );
        break;

    case 'O':
    case 'o':
        len = strlen(id+1);
        if ( len && len + 5 < size )
        {
            buf[0] = '?';
            buf[1] = 'O';
            buf[2] = '=';
            if ( len )
                memcpy( buf+3, id+1, len );
            return len+3;
        }
        return 0;

    case '*':
        off = 0;

        while (( dli.fi = (*dl->dir_read)( param ) ))
        {
            if ( dli.fi->mtime != 0 && dli.fi->mtime != (time_t)(-1) )
                dli.ptm = localtime( &(dli.fi->mtime) );
            else
                dli.ptm = 0;

            switch ( dli.fi->ft )
            {
            case FT_DIR:
            case FT_ZIPDIR:
                templ = CONFIG_OF(index_templ_dir);
                break;
            case FT_FILE:
            case FT_ZIPFILE:
                templ = CONFIG_OF(index_templ_file);
                break;
            case FT_ZIP:
                templ = CONFIG_OF(index_templ_zip);
                break;
            default:
                templ = CONFIG_OF(index_templ_other);
                break;
            }

            off += html_from_template( buf+off, size-off, templ,
                                       html_dir_list_item_callback, &dli );
            if ( off >= size -1 )
                break;
        }
        break;

    default:
        if ( size <= 4 )
            return 0;
        memcpy( buf, "N/A", 3 );
        off = 3;
        break;
    }

    return off;
}

memblock *html_dir_list_ex( struct s_dir_list *dl )
{
    size_t len;
    memblock *m = memblock_new( PREALLOC_SIZE );
    if ( !m )
        return 0;

    len = html_from_template( m->ptr, m->size, CONFIG_OF(index_templ),
                              html_dir_list_callback, dl );

    return memblock_realloc( m, len );
}



/* generate HTML dir list directly from the disk */

static fileinfo *_func_dir_read( void *param )
{
    static fileinfo fi;
    if ( dir_read( ((struct s_dir_list *)param)->p1.pdir, &fi ) >= 0 )
    {
        if ( fi.ft == FT_FILE )
        {
            const char *p = strstr(fi.name, ".zip" );
            if ( !p )
                p = strstr( fi.name, ".ZIP" );
            if ( p && '\0' == p[4] )
                fi.ft = FT_ZIP;
        }
        return &fi;
    }
    return 0;
}

memblock *html_dir_list( const char *dir_name, const char *uri )
{
    memblock *m;
    directory di;
    struct s_dir_list dl;

    if ( dir_open( &di, dir_name ) < 0 )
        return 0;

    dl.uri = uri;
    dl.p1.pdir = &di;
    dl.dir_read = _func_dir_read;
    m = html_dir_list_ex( &dl );

    dir_close( &di );

    return m;
}

/* generate HTML dir list directly from fileinfo array with a sort option */

static const char *_fileinfo_sort_option;

static int _func_fileinfo_compare( const void *p1, const void *p2 )
{
    int i, n;
    fileinfo *fi1, *fi2;

    fi1 = *(fileinfo **)p1;
    fi2 = *(fileinfo **)p2;

    for ( i=0; _fileinfo_sort_option[i]; i++ )
    {
        switch ( _fileinfo_sort_option[i] )
        {
        case SORT_BY_DIR_A:
            if ( fi1->ft == fi2->ft )
                continue;
            return (int)( fi1->ft - fi2->ft );
        case SORT_BY_DIR_D:
            if ( fi1->ft == fi2->ft )
                continue;
            return (int)( fi2->ft - fi1->ft );
        case SORT_BY_NAME_A:
            n = strcmp( fi1->name, fi2->name );
            if ( 0 == n ) continue;
            return n;
        case SORT_BY_NAME_D:
            n = strcmp( fi1->name, fi2->name );
            if ( 0 == n ) continue;
            return -n;
        case SORT_BY_SIZE_A:
            if ( fi1->size == fi2->size )
                continue;
            return (int)fi1->size - (int)fi2->size;
        case SORT_BY_SIZE_D:
            if ( fi1->size == fi2->size )
                continue;
            return (int)fi2->size - (int)fi1->size;
        case SORT_BY_DATE_A:
            if ( fi1->mtime == fi2->mtime )
                continue;
            return (int)fi1->mtime - (int)fi2->mtime;
        case SORT_BY_DATE_D:
            if ( fi1->mtime == fi2->mtime )
                continue;
            return (int)fi2->mtime - (int)fi1->mtime;
        default:
            break;
        }
    }

    return 0;
}

static fileinfo *_func_fileinfo_read( void *param )
{
    struct s_dir_list *dl = param;
    return dl->p1.fi_index[dl->p2.at++];
}

memblock *html_dir_list_sorted( fileinfo *fi_dir, const char *uri,
                                const char *so )
{
    struct s_dir_list dl;
    int i;
    memblock *m;

    dl.p1.fi_index = malloc( MAX_DIR_ENTRYNUM * sizeof(fileinfo *) );
    if ( !dl.p1.fi_index )
        return 0;

    for ( i=0; fi_dir[i].ft != FT_NONE; i++ )
        dl.p1.fi_index[i] = fi_dir+i;

    assert( i < MAX_DIR_ENTRYNUM );
    dl.p1.fi_index[i] = 0;

    if ( i )
    {
        _fileinfo_sort_option = so;
        qsort( dl.p1.fi_index, i, sizeof(dl.p1.fi_index[0]),
               _func_fileinfo_compare );
    }

    dl.uri = uri;
    dl.p2.at = 0;
    dl.dir_read = _func_fileinfo_read;
    m = html_dir_list_ex( &dl );

    free( dl.p1.fi_index );

    return m;
}


/* generate HTML dir list from unsorted dir in a zipfile */

static fileinfo *_func_zipdir_read( void *param )
{
    char *p;
    static fileinfo fi;
    struct s_dir_list *dl = param;

    if ( 0 == dl->p2.ze_at )
    {
        if ( 0 == dl->p1.ze_head )
            return 0;
        dl->p2.ze_at = dl->p1.ze_head;
    }
    else if ( dl->p2.ze_at == dl->p1.ze_head )
        return 0;

    fi.ctime = fi.atime = fi.mtime = dl->p2.ze_at->time;
    fi.name = dl->p2.ze_at->name;

    p = strrchr( fi.name, '/' );
    if ( p )
        fi.name = p+1;

    fi.size = dl->p2.ze_at->un_size;
    fi.ft = dl->p2.ze_at->files ? FT_ZIPDIR : FT_ZIPFILE;

    dl->p2.ze_at = dl->p2.ze_at->next;
    return &fi;
}

memblock *html_zipdir_list( zipentry *first, const char *uri )
{
    struct s_dir_list dl;

    dl.uri = uri;
    dl.p1.ze_head = first;
    dl.p2.ze_at = 0;
    dl.dir_read = _func_zipdir_read;
    return html_dir_list_ex( &dl );
}

/* generate HTML dir list from sorted dir in a zipfile */

static const char *_zipentry_sort_option;

static int _func_zipentry_compare( const void *p1, const void *p2 )
{
    int i, n, nr;
    zipentry *ze1, *ze2;

    ze1 = *(zipentry **)p1;
    ze2 = *(zipentry **)p2;

    for ( i=0; _zipentry_sort_option[i]; i++ )
    {
        switch ( _zipentry_sort_option[i] )
        {
        case SORT_BY_DIR_A:
            n = ze1->file_num & !ze2->file_num;
            nr = !ze1->file_num & ze2->file_num;
            if ( n == nr ) continue;
            return (int)( n - nr );
        case SORT_BY_DIR_D:
            n = ze1->file_num & !ze2->file_num;
            nr = !ze1->file_num & ze2->file_num;
            if ( n == nr ) continue;
            return (int)( nr - n );
        case SORT_BY_NAME_A:
            n = strcmp( ze1->name, ze2->name );
            if ( 0 == n ) continue;
            return n;
        case SORT_BY_NAME_D:
            n = strcmp( ze1->name, ze2->name );
            if ( 0 == n ) continue;
            return -n;
        case SORT_BY_SIZE_A:
            if ( ze1->un_size == ze2->un_size )
                continue;
            return (int)ze1->un_size - (int)ze2->un_size;
        case SORT_BY_SIZE_D:
            if ( ze1->un_size == ze2->un_size )
                continue;
            return (int)ze2->un_size - (int)ze1->un_size;
        case SORT_BY_DATE_A:
            if ( ze1->time == ze2->time )
                continue;
            return (int)ze1->time - (int)ze2->time;
        case SORT_BY_DATE_D:
            if ( ze1->time == ze2->time )
                continue;
            return (int)ze2->time - (int)ze1->time;
        default:
            break;
        }
    }

    return 0;
}


static fileinfo *_func_zipdir_read_s( void *param )
{
    char *p;
    static fileinfo fi;
    struct s_dir_list *dl = param;

    if ( ! dl->p1.ze_index[dl->p2.at] )
        return 0;

    fi.ctime = fi.atime = fi.mtime = dl->p1.ze_index[dl->p2.at]->time;
    fi.name = dl->p1.ze_index[dl->p2.at]->name;

    p = strrchr( fi.name, '/' );
    if ( p )
        fi.name = p+1;

    fi.size = dl->p1.ze_index[dl->p2.at]->un_size;

    fi.ft = dl->p1.ze_index[dl->p2.at]->files ? FT_ZIPDIR : FT_ZIPFILE;
    dl->p2.at++;
    return &fi;
}

memblock *html_zipdir_list_sorted( zipentry *first, const char *uri,
                                   const char *so )
{
    struct s_dir_list dl;
    zipentry *p;
    int i;
    memblock *m;

    dl.p1.ze_index = malloc( MAX_DIR_ENTRYNUM * sizeof(zipentry *) );
    if ( !dl.p1.ze_index )
        return 0;

    i = 0;
    if ( first )
    {
        p = first;
        do
        {
            dl.p1.ze_index[i] = p;
            i++;
            if ( i >= MAX_DIR_ENTRYNUM - 1 )
                break;
            p = p->next;
        }
        while( p != first );

        _zipentry_sort_option = so;
        qsort( dl.p1.ze_index, i, sizeof(dl.p1.ze_index[0]),
               _func_zipentry_compare );
    }

    dl.p1.ze_index[i] = 0;

    dl.uri = uri;
    dl.p2.ze_at = 0;
    dl.dir_read = _func_zipdir_read_s;
    m = html_dir_list_ex( &dl );
    free( dl.p1.ze_index );

    return m;
}

/* Debugging functions */


#ifdef DEBUG_HTMLGEN
int main( int argc, char *argv[] )
{
    memblock *m;

    if ( argc < 2 )
    {
        printf( "Usage:\n\t%s <dir>\n", argv[0] );
        return 0;
    }

    CONFIG_INIT();
    validate_config( CONFIG );
    m = html_dir_list( argv[1], argc >= 3 ? argv[2] : argv[1] );
    if ( m )
        puts( m->ptr );
    else
        fprintf( stderr, "Failed.\n" );
    memblock_delete( m );
    CONFIG_CLEANUP();

    return 0;
}
#endif
