/*
 * htconfig.c: web server configuration
 *
 * Hanhua Feng
 * $Id: htconfig.c,v 1.6 2003/05/21 14:59:39 hanhua Exp $
 */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "htutil.h"
#include "htconfig.h"
#include "rc_template.h"


htconfig config;

#define LARGE_BUF_SIZE 32768
#define MIN_VALUE_BUF 64

char *family_name[FAMILY_NUM] = {
    "httpd", "directory", "index"
};

configentry *create_config_entry( htconfig *conf, const char *name,
        int family, const char *value, size_t len )
{
    int name_len;
    configentry *ce;

    name_len = strlen(name);

    if ( len < MIN_VALUE_BUF )
        len = MIN_VALUE_BUF;

    ce = htmalloc( CONFIG_STORAGE, sizeof(configentry) + name_len + len + 2 );
    if ( !ce )
        return 0;

    ce->name = (char *)(void *)(ce+1);
    ce->value = ce->name + name_len + 1;
    ce->rname = 0;
    ce->def = 0;

    strcpy( ce->name, name );
    str_to_lower( ce->name );
    strcpy( ce->value, value );
    ce->family = family;

    HASH_ADD( *conf, ce->name, ce );
    DLINKED_ADD( ce, conf->family[family] );

    return ce;
}

configentry *create_def_config_entry( htconfig *conf, const char *name,
        int family, const char *def, size_t len )
{
    configentry *ce = create_config_entry( conf, name, family, "", 0 );
    if ( !def )
        ce->def = ce->value;
    else
        ce->def = def;
    ce->rname = name;
    return ce;
}

void delete_config_entry( htconfig *conf, configentry *ce )
{
    DLINKED_REMOVE( ce, conf->family[ce->family] );
    HASH_REMOVE( *conf, ce->name, ce );
    htfree( CONFIG_STORAGE, ce );
}

int load_config_file( htconfig *conf, const char *file )
{
    char buf[4096], *p, *pt;
    size_t p_len;
    int i, fam=-1, count = 0;
    char *large_buf = 0;
    FILE *fp;

    fp = htfopen( file, "rt" );
    if ( !fp )
        return -1;

    while( fgets( buf, sizeof(buf), fp ) )
    {
        p = buf;
        while ( isascii(*p) && isspace(*p) ) p++;

        /* assuming the comment begins with ; # % */
        if ( ';' == *p || '#' == *p || '%' == *p || '\0' == *p )
            continue;

        if ( '[' == *p )
        {
            /* a family name */
            p++;

            pt = strchr( p, ']' );
            if ( !pt )
            {
                count = -1;
                break;
            }
            *pt = '\0';

            p = str_to_lower( str_trim( p ) );

            fam = -1;
            for ( i=0; i<FAMILY_NUM; i++ )
                if ( 0 == strcmp( family_name[i], p ) )
                    fam = i;
        }
        else if ( fam <= 0 )
            continue;
        else
        {
            /* format: name=value */
            p = strchr( buf, '=' );
            if ( !p )
            {
                count = -1;
                break;
            }

            *p++ = '\0';

            pt = str_to_lower( str_trim(buf) );
            p = str_trim(p);


            p_len = strlen(p);

            /* multi-line value begins with name = @@ and ends with @@ */
            if ( p[0] == '@' && p[1] == '@' )
            {
                if ( !large_buf )
                {
                    large_buf = malloc( LARGE_BUF_SIZE );
                    if ( !large_buf )
                    {
                        count = -1;
                        break;
                    }
                }

                p_len = 0;
                while ( fgets( large_buf + p_len,
                               LARGE_BUF_SIZE - p_len, fp ) )
                {
                    p = strstr( large_buf + p_len, "@@" );
                    if ( p )
                    {
                        *p = '\0';
                        break;
                    }
                    p_len += strlen( large_buf + p_len );
                    if ( p_len >= LARGE_BUF_SIZE -2 )
                        break;
                }

                p = large_buf;
            }

            /* creating the hash entry */
            create_config_entry( conf, pt, fam, p, p_len );
            count++;
        }
    }

    if ( large_buf )
        free( large_buf );

    htfclose( fp );

    return count;
}

static void _save_config_item( FILE *fp, configentry *ce )
{
    if ( ce->def )
        return;

    if ( strchr( ce->value, '\n' ) )
    {
        fprintf( fp, "%s=@@\n", ce->rname ? ce->rname : ce->name );
        fputs( ce->value, fp );
        fputs( "@@\n", fp );
    }
    else
        fprintf( fp, "%s=%s\n",
                 ce->rname ? ce->rname : ce->name, ce->value );
}

int save_config_file( htconfig *conf, const char *file )
{
    FILE *fp;
    int i;
    configentry *ce;

    fp = htfopen( file, "rt" );
    if ( !fp )
        return -1;

    for ( i=0; i<FAMILY_NUM; i++ )
    {
        fprintf( fp, "[%s]\n", file );
        DLINKED_ITERATE( conf->family[i], ce, (_save_config_item(fp,ce)) );
    }

    htfclose( fp );
    return 0;
}

static int multi_line_to_str_array( const char **array, int num, char *buf )
{
    size_t len;
    int i;
    char *p, *pt, *pw;

    for ( i=0, p=pw=buf; i<num-1 && p; p=pt )
    {
        pt = strchr( p, '\n' );
        if ( !pt )
            len = strlen(p);
        else
        {
            len = pt - p;
            *pt = '\0';
        }
        array[i] = str_trim( p );
        if ( array[i][0] )
        {
            if ( array[i] != pw )
                memmove( pw, array[i], len );
            pw[len++] = '\n';
            array[i++] = pw;
            pw += len;
        }
    }
    array[i] = 0;

    return i;
}

configentry *find_config_entry( htconfig *conf,
                                const char *name, int family )
{
    configentry *ce;

    HASH_FIND(*conf,name,ce);
    while( ce )
    {
        if ( ce->family == family )
            break;
        HASH_FIND_NEXT(*conf,name,ce);
    }

    return ce;
}

int validate_int_config( htconfig *conf, const char *name, int family,
                         int def, int min, int max )
{
    int n;
    configentry *ce = find_config_entry( conf, name, family );

    if ( ce )
    {
        if ( !ce->rname )
            ce->rname = name;

        n = atoi( ce->value );
        if ( n >= min && n <= max )
            return n;
        else
            sprintf( ce->value, "%d", def );
    }
    else
    {
        ce = create_def_config_entry( conf, name, family, 0, 0 );
        sprintf( ce->value, "%d", def );
    }

    return def;
}

const char *validate_str_config( htconfig *conf, const char *name, int family,
                                 const char *def )
{
    configentry *ce = find_config_entry( conf, name, family );
    if ( ce )
    {
        if ( !ce->rname )
            ce->rname = name;
        return ce->value;
    }

    if ( def )
        create_def_config_entry( conf, name, family, def, strlen(def) );
    return def;
}

int validate_str_array_config( htconfig *conf, const char *name, int family,
                               const char **array, int num, const char *def )
{
    configentry *ce = find_config_entry( conf, name, family );
    if ( ce )
    {
        if ( !ce->rname )
            ce->rname = name;
        return multi_line_to_str_array( array, num, ce->value );
    }

    if ( def )
    {
        assert( num >= 2 );
        ce = create_def_config_entry( conf, name, family, def, strlen(def) );
        array[0] = ce->def;
        array[1] = 0;
    }

    return 0;
}

void validate_httpd_config( htconfig *conf )
{
    conf->server_port = validate_int_config(
        conf, TAG_SERVER_PORT, FAM_HTTPD, 80, 1, 32760 );
    conf->conn_timeout = validate_int_config(
        conf, TAG_CONNECTION_TIMEOUT, FAM_HTTPD, 10, 4, 900 );
    conf->server_name = validate_str_config(
        conf, TAG_SERVER_NAME, FAM_HTTPD, "?The serverName" );
}


void validate_dir_config( htconfig *conf, const char *work_dir )
{
    validate_str_array_config(
            conf, TAG_URI_MAPPING, FAM_DIR, conf->uri_mapping,
            sizeof(conf->uri_mapping)/sizeof(conf->uri_mapping[0]),
            work_dir );
    validate_str_array_config(
            conf, TAG_ALLOW_PATH, FAM_DIR, conf->allow_path,
            sizeof(conf->allow_path)/sizeof(conf->allow_path[0]),
            work_dir );
}

void validate_index_config( htconfig *conf )
{
    conf->index_file = validate_str_config(
        conf, TAG_INDEX_FILE, FAM_INDEX, "index.html" );
    conf->readme_file = validate_str_config(
        conf, TAG_README_FILE, FAM_INDEX, "README" );
    conf->index_templ = validate_str_config(
        conf, TAG_INDEX_TEMPLATE, FAM_INDEX, RC_GET(index_templ) );
    conf->index_templ_file = validate_str_config(
        conf, TAG_INDEX_TEMPLATE_FILE, FAM_INDEX,
        RC_GET(index_templ_file) );
    conf->index_templ_dir = validate_str_config(
        conf, TAG_INDEX_TEMPLATE_DIR, FAM_INDEX,
        RC_GET(index_templ_dir) );
    conf->index_templ_other = validate_str_config(
        conf, TAG_INDEX_TEMPLATE_OTHER, FAM_INDEX,
        RC_GET(index_templ_other) );
    conf->index_templ_zip = validate_str_config(
        conf, TAG_INDEX_TEMPLATE_ZIP, FAM_INDEX,
        RC_GET(index_templ_zip) );
}

void validate_config( htconfig *conf, const char *work_dir )
{
    validate_httpd_config( conf );
    validate_dir_config( conf, work_dir );
    validate_index_config( conf );
}
