/*
 * file.c: processing different kinds of files
 *
 * Hanhua Feng
 * $Id: file.c,v 1.12 2003/05/21 14:59:39 hanhua Exp $
 */
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "htutil.h"
#include "zip.h"
#include "htzipd.h"

/* The structure fileentry is very complex. It knows what storage is managed
 * by itself and what is not.
 * Data field:
 *    count: the access count. If it is zero, the fileentry structure can be
 *           released.
 *    ft:    the type of entry.
 *       -- FT_ZIP: a zip file. ( using zip )
 *       -- FT_DIR: a preloaded dir ( using fi )
 *       -- FT_MEMORY: a memory block ( using mem )
 *       -- FT_STATIC: a non-releasable pointer (using cp)
 *       -- FT_FILE: a FILE pointer ( using fp )
 *       -- FT_ZIPDIR: a zipentry pointer ( using ze )
 *       -- FT_ZIPFILE: a zipsession pointer ( using zs )
 *       -- otherwise: invalid.
 *    time:  the time of information (File/dir modification time, etc)
 *    size:  the size of information. 0 if invalid.
 *    p.p:   general pointer
 *    p.fp:  a FILE pointer. Using fclose to release resources.
 *    p.mem: a memblock structure. If not static, using memblock_delete to
 *           release resources.
 *    p.zip: a zipfile structure. Using zip_close to release resources.
 *    p.fi:  a fileinfo array. Everything is in a single block, just use
 *           free() to release resources.
 *    p.ze:  a zipentry. Need not to release resources.
 *    p.zs:  a zipsession. Using zip_session_close to release resources.
 *    base:  The base fileentry structure. Valid for FT_ZIPDIR and FT_ZIPFILE.
 */
void fileentry_delete_object( enum FileType ft, void *p )
{
    switch ( ft )
    {
    case FT_ZIP:
        zip_close( (zipfile *)p );
        break;
    case FT_MEMORY:
        memblock_delete( p );
        break;
    case FT_FILE:
        htfclose( (FILE *)p );
        break;
    case FT_DIR:
        htfree( FILEINFO_STORAGE, p );
        break;
    case FT_ZIPFILE:
        zip_session_close( (zipsession *)p );
        break;
    case FT_ZIPDIR:
    case FT_STATIC:
    default:
        break;
    }
}

static fileentry *fileentry_new( enum FileType ft, const char *name,
                                 void *p, time_t t )
{
    fileentry *fe;
    int n = strlen(name) + 1;

    fe = htmalloc( FILE_STORAGE, sizeof(fileentry) + n );
    if ( !fe )
    {
        fileentry_delete_object( ft, p );
        return 0;
    }

    fe->count = 0;

    fe->ft = ft;
    fe->name = (char *)(void *)(fe+1);
    strcpy( fe->name, name );
    fe->p.p = p;
    fe->time = t;
    fe->base = 0;

    switch ( ft )
    {
    case FT_DIR:
        fe->size = 0;
        break;
    case FT_STATIC:
        assert(0); /* we should call fileentry_static_init instead */
        break;
    case FT_MEMORY:
        fe->size = ((memblock*)p)->size;
        break;
    case FT_ZIP:
        fseek( ((zipfile *)p)->fp, 0, SEEK_END );
        fe->size = ftell( ((zipfile *)p)->fp );
        break;
    case FT_FILE:
        fseek( (FILE *)p, 0, SEEK_END );
        fe->size = ftell( (FILE *)p );
        break;
    default:
        break;
    }

    return fe;
}

static fileentry *fileentry_new_on( enum FileType ft, const char *name,
                                    void *p, time_t t, fileentry *base )
{
    fileentry *fe = fileentry_new( ft, name, p, t );
    if ( !fe )
        return 0;

    if ( base )
    {
        fe->base = base;
        fileentry_inc( fe->base );
    }
    return fe;
}

static fileentry *fileentry_new_file_in_zip( fileentry *base, zipsession *zs,
                                             const char *name )
{
    fileentry *fe = fileentry_new_on(
            FT_ZIPFILE, name, zs, zs->entry->time, base );
    fe->size = zs->entry->un_size;
    return fe;
}


void fileentry_delete( fileentry *fe )
{
    if ( fe->base )
        fileentry_dec( fe->base );

    fileentry_delete_object( fe->ft, fe->p.p );
    htfree( FILE_STORAGE, fe );
}

void fileentry_inc( fileentry *fe )
{
    fe->count++;
}

void fileentry_dec( fileentry *fe )
{
    fe->count--;
}


fileentry *fileentry_static_init( htman *man, char *name,
                                  const char *buf, size_t size )
{
    fileentry *fe;

    fe = htmalloc( FILE_STORAGE, sizeof(fileentry) );
    if ( !fe )
        return 0;
    fe->ft = FT_STATIC;
    fe->size = size;
    fe->p.cp = buf;
    fe->time = 0;
    fe->count = 1;
    fe->base = 0;

    fe->name = name;

    HASH_ADD( man->files, fe->name, fe );

    return fe;
}

fileentry *fileentry_memblock_init( htman *man, char *name, memblock *m )
{
    fileentry *fe;
    size_t len = strlen( name );

    if ( !m )
        return 0;

    fe = htmalloc( FILE_STORAGE, sizeof(fileentry) + len + 1 );
    if ( !fe )
        return 0;
    fe->ft = FT_MEMORY;
    fe->size = m->size;
    fe->p.mem = m;
    fe->time = 0;
    fe->count = 0;
    fe->base = 0;

    fe->name = (void *)(fe+1);
    strcpy( fe->name, name );

    HASH_ADD( man->files, fe->name, fe );

    return fe;
}

fileentry *fileentry_init( htman *man, const char *path, const char *uri,
                           const char *query, int *f_app_slash )
{
    static char s_path[MAX_PATH + 256];
    fileentry *fe;
    zipentry *ze;
    zipsession *zs;
    zipfile *zf;
    int len, zip_len, dir_flag;
    enum FileType ft;
    char *p;
    FILE *fp;
    fileinfo fi;

    *f_app_slash = 0;

    /* path can't be an empty string */
    if ( *path == '\0' )
        return 0;

    /* copy path to our own buffer */
    len = strlen(path);
    if ( len >= sizeof(s_path)-2 )
        return 0;
    strcpy( s_path, path );

    /* Has this file already been cached in man->files? */
    if ( s_path[len-1] != '/' || !query )
    {
        HASH_FIND( man->files, path, fe );
        if ( fe )
            return fe;
    }

    /* Has this file + '/' already been cached in man->files? */
    if ( s_path[len-1] != '/' )   /* len must be >= 1 */
    {
        s_path[len] = '/';
        s_path[len+1] = '\0';

        HASH_FIND( man->files, path, fe );
        if ( fe )
        {
            *f_app_slash = 1;
            return fe;
        }

        s_path[len] = '\0';
        dir_flag = 0;
    }
    else
    {
        dir_flag = 1;

        if ( query && query[0] == 'O' && query[1] == '=' )
        {
            if ( strlen(query) + len >= sizeof(s_path) - 1 )
                return 0;
            s_path[len] = '*';
            strcpy( s_path+len+1, query+2 );

            HASH_FIND( man->files, s_path, fe );
            if ( fe )
                return fe;

            s_path[len] = '\0';
        }
    }


    /* Now we know this file has never been cached */

    /* belongs to a zip file? (the dir of zipfile may have been cached) */
    for ( p = strchr( s_path, '.' ); p; p = strchr( p, '.' )  )
    {
        if ( tolower(p[1]) == 'z' && tolower(p[2]) == 'i'
             && tolower(p[3]) == 'p' && p[4] == '/' )
        {
            /* the path contains a ".zip/" sub-string */
            p += 4;

            /* substitute '/' with '\0' */
            zip_len = p - s_path;
            s_path[zip_len] = '\0';
            p++;

            HASH_FIND( man->files, s_path, fe );
            if ( fe )
            {
                switch ( fe->ft )
                {
                case FT_FILE:
                    /* If the zip file has been accessed as a whole */
                    if ( 0 == fe->count )
                    {
                        HASH_REMOVE( man->files, s_path, fe );
                        fileentry_delete( fe );
                    }
                    fe = 0;
                    break;
                case FT_ZIP:
                    zf = fe->p.zip;
                    break;
                case FT_MEMORY:
                case FT_STATIC:
                    continue;
                default:
                    return 0;
                }
            }

            if ( !fe )
            {
                /* The zipfile has not been openned yet */
                ft = get_file_info( s_path, &fi );
                switch ( ft )
                {
                case FT_FILE: /* It's a file, so it could be a zip file */

                    /* open/check the zip file */
                    zf = zip_open( s_path );
                    if ( !zf )
                        return 0;   /* damaged zip file */

                    while( zip_loadentry( zf, 1000 ) == 1000 )
                        ;    /* load all zip entries */

                    /* add new zipfile to man->files. no trail slash */
                    fe = fileentry_new( FT_ZIP, s_path, zf, fi.mtime );
                    if ( !fe )
                    {
                        zip_close( zf );
                        return 0;
                    }

                    HASH_ADD( man->files, s_path, fe );
                    break;

                case FT_DIR:
                    /* It's a dir with .zip postfix, not a zipfile so we
                     * change '\0' back to '/' */
                    s_path[zip_len] = '/';
                    continue;

                default:
                    /* In other cases, we needn't check it deeper. */
                    return 0;
                }
            }

            /* Now the zipfile has been found or added in man->files
             * and p should point to an inner path without prepended '/'.
             * Meanwhile p[-1] should be '\0'. then we try to remove
             * appended slashes assuming that double slashes have been
             * removed. s_path could be equal to p. */
            if ( '/' == s_path[len-1] )
                s_path[--len] = '\0';

            if ( dir_flag )
            {
                /* It should be a dir in the zipfile */
                ze = zip_readdir( zf, p );
                if ( !ze )
                    return 0;
            }
            else
            {
                /* searching the file in the zipfile */
                ze = zip_search( zf, p );
                if ( !ze )
                    return 0;

                if ( ze->files )
                {
                    /* It's actuallly a dir, but '/' is not appended */
                    ze = ze->files;
                    dir_flag = 1;
                    *f_app_slash = 1;
                    s_path[len++] = '/';
                    s_path[len] = '\0';
                }
                else
                {
                    zs = zip_session_open( zf, ze );
                    if ( !zs )
                        return 0;

                    fe = fileentry_new_file_in_zip( fe, zs, path );
                    HASH_ADD( man->files, fe->name, fe );
                }
            }

            /* generate the dir list of zipfile into memory */
            if ( dir_flag )
            {
                if ( query && query[0] == 'O' && query[1] == '=' )
                {
                    strcpy( s_path, path );
                    len = strlen( s_path );
                    s_path[len++] = '*';
                    strcpy( s_path+len, query+2 );

                    fe = fileentry_new(
                        FT_MEMORY, s_path,
                        html_zipdir_list_sorted( ze, uri, query+2 ),
                        fi.mtime );
                }
                else
                {
                    fe = fileentry_new(
                        FT_MEMORY, path,
                        html_zipdir_list( ze, uri ), ze->time );
                }
                HASH_ADD( man->files, fe->name, fe );
            }
            return fe;
        }

        p++;
    }

    /* Now it's neither in cache nor a zip file. */

    ft = get_file_info( path, &fi );
    switch ( ft )
    {
    case FT_DIR:
        if ( !dir_flag )
        {
            s_path[len++] = '/';
            s_path[len] = '\0';
            *f_app_slash = 1;
            return 0;
        }

        if ( query && query[0] == 'O' && query[1] == '=' )
        {
            s_path[len-1] = '*';
            s_path[len] = '\0';
            HASH_FIND( man->files, s_path, fe );
            if ( !fe )
            {
                fileinfo *fi_array;
                s_path[len-1] = '\0';

                fi_array = dir_load(s_path);
                if ( !fi_array )
                    return 0;

                s_path[len-1] = '*';
                fe = fileentry_new( FT_DIR, s_path, fi_array, fi.mtime );
                if ( !fe )
                {
                    htfree( FILEINFO_STORAGE, fi_array );
                    return 0;
                }

                HASH_ADD( man->files, fe->name, fe );
            }

            /* supposing string s_path+len+1 has not been changed */
            s_path[len-1] = '/';
            s_path[len] = '*';

            fe = fileentry_new(
                FT_MEMORY, s_path,
                html_dir_list_sorted( fe->p.fi, uri, query+2 ),
                fi.mtime );

            break;
        }

        fe = fileentry_new( FT_MEMORY, s_path,
                            html_dir_list(path,uri), fi.mtime );
        break;
    case FT_FILE:
        if ( dir_flag )
            return 0;
        fp = htfopen( path, "rb" );
        fe = fileentry_new( FT_FILE, path, fp, fi.mtime );
        break;
    default:
        return 0;
    }

    HASH_ADD( man->files, fe->name, fe );
    return fe;
}

int fileentry_load( fileentry *fe, char *buf, size_t size, size_t offset )
{
    zipfile *zf;
    zipsession *zs;

    assert( size+offset <= fe->size );

    switch ( fe->ft )
    {
    case FT_FILE:
        fseek( fe->p.fp, offset, SEEK_SET );
        fread( buf, 1, size, fe->p.fp );
        break;
    case FT_ZIP:
        fseek( fe->p.zip->fp, offset, SEEK_SET );
        fread( buf, 1, size, fe->p.zip->fp );
        break;
    case FT_STATIC:
        memcpy( buf, fe->p.cp + offset, size );
        break;
    case FT_MEMORY:
        memcpy( buf, fe->p.mem->ptr + offset, size );
        break;
    case FT_ZIPFILE:
        assert( fe->base );
        zf = fe->base->p.zip;
        zs = fe->p.zs;
        if ( !zf || !zs )
            return REQ_ERR;
        if ( zs->offset != offset )
            zip_session_seek( zs, offset );
        if ( size != zip_session_read( buf, size, zs ) )
            return REQ_ERR;
        break;
    default:
        return REQ_ERR;
    }

    return REQ_OK;
}
