/*
 * error.c: error processing
 *
 * Hanhua Feng
 * $Id: error.c,v 1.6 2003/05/21 14:59:39 hanhua Exp $
 */
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "htutil.h"
#include "htconfig.h"
#include "htzipd.h"
#include "rc_template.h"

static const char *err_stat_table[HTTP_STATUS_NUM];
static const char *err_msg_table[HTTP_STATUS_NUM];

void error_response_init( htman *man )
{
    err_stat_table[HTTP302] = "302 Found";
    err_msg_table[HTTP302] = RC_GET(http_302);

    err_stat_table[HTTP400] = "400 Bad request";
    err_msg_table[HTTP400] = RC_GET(http_400);

    err_stat_table[HTTP403] = "403 Forbidden";
    err_msg_table[HTTP403] = RC_GET(http_403);

    err_stat_table[HTTP404] = "404 Not found";
    err_msg_table[HTTP404] = RC_GET(http_404);

    err_stat_table[HTTP405] = "405 Method not allowed";
    err_msg_table[HTTP405] = RC_GET(http_405);

    err_stat_table[HTTP413] = "413 Request engity too large";
    err_msg_table[HTTP413] = RC_GET(http_413);

    err_stat_table[HTTP414] = "414 Request-URI too long";
    err_msg_table[HTTP414] = RC_GET(http_414);

    err_stat_table[HTTP416] = "416 Requested Range Not Satisfiable";
    err_msg_table[HTTP416] = RC_GET(http_416);

    err_stat_table[HTTP500] = "500 Internal server error";
    err_msg_table[HTTP500] = RC_GET(http_500);

    err_stat_table[HTTP501] = "501 Not implemented";
    err_msg_table[HTTP501] = RC_GET(http_501);

    err_stat_table[HTTP505] = "505 Version not implemented";
    err_msg_table[HTTP505] = RC_GET(http_505);
}

static size_t _func_err_msg( char *buf, size_t buf_size, const char *templ,
                             void *param )
{
    const char *p;
    size_t len;
    taskentry *tsk = param;

    switch( *templ )
    {
    case 'u':
        p = strrchr( tsk->request_uri, '/' );
        if ( ! p[1] )
        {
            for ( ; p > tsk->request_uri; p-- )
                if ( '/' == p[-1] )
                    break;
        }
        if ( p )
            p++;
        else
            p = tsk->request_uri;

        len = strlen( p );
        if ( len >= buf_size - 1 )
            return 0;
        memcpy( buf, p, len );
        break;

    case 'U':  /* uri full path */
        len = strlen( tsk->request_uri );
        if ( len >= buf_size - 1 )
            return 0;
        memcpy( buf, tsk->request_uri, len );
        break;

    case 'V':
        if ( buf_size < 32 )
            return 0;
        len = sprintf( buf, "%d.%d", tsk->http_version>>8,
                        tsk->http_version & 0xff );
        break;

    case 'L':
        len = strlen(tsk->new_location);
        if ( len >= buf_size -1 )
            return 0;
        memcpy( buf, tsk->new_location, len );
        break;

    default:
        len = 0;
        break;
    }

    return len;
}

size_t generate_error_response( taskentry *tsk,
    char *buf, size_t buf_size, int err_code, int f_entity )
{
    char str[8];
    size_t off, off_num, len;

    assert( err_code > 0 && err_code < HTTP_STATUS_NUM );

    if ( 0 == err_stat_table[err_code] )
    {
#ifdef DEBUG
        fprintf( stderr,
                 "Error code %d can not be found in the message table.\n",
                 err_code );
#endif
        return 0;
    }

    off = sprintf( buf, "HTTP/1.1 %s\r\n", err_stat_table[err_code] );
    switch ( err_code )
    {
    case HTTP302:
        off += sprintf( buf+off, "Location: %s\r\n", tsk->new_location );
        break;
    default:
        break;
    }

    if ( f_entity )
    {
        off += sprintf( buf+off,
            "Content-Type: text/html\r\nContent-Length: " );
        off_num = off;
        off += 4;
        off += sprintf( buf+off, "\r\n\r\n" );

        len = html_from_template( buf+off, buf_size-off,
            err_msg_table[err_code], _func_err_msg, tsk );
        off += len;

        sprintf( str, "%4d", len );
        memcpy( buf+off_num, str, 4 );
    }
    else
        off += sprintf( buf+off, "\r\n" );

    return off;
}
