#ifndef _GLLIGHT_H
#define _GLLIGHT_H

#include <gl\gl.h>										// Header File For The OpenGL32 Library
#include <gl\glu.h>										// Header File For The GLu32 Library
#include "glvector.h"

class CFunLight
{
public:
	float position[4];

	CFunLight(GLenum n)
	{
		id = n;
	}

	void Reset()
	{
		position[0] = 0;
		position[1] = 0;
		position[2] = 1;
		position[3] = 0;

		ambient[0] = 0;
		ambient[1] = 0;
		ambient[2] = 0;
		ambient[3] = 1;

		diffuse[0] = 1;
		diffuse[1] = 1;
		diffuse[2] = 1;
		diffuse[3] = 1;

		specular[0] = 1;
		specular[1] = 1;
		specular[2] = 1;
		specular[3] = 1;

		spotlight = false;
		spotExponent = 0;
		spotCutOff = 180;
		spotDirection.Set(0, 0, -1);
		attenuation.Set(1, 0, 0);

		glLightfv(id, GL_POSITION, position);
		glLightfv(id, GL_AMBIENT, ambient);
		glLightfv(id, GL_DIFFUSE, diffuse);
		glLightfv(id, GL_SPECULAR, specular);
		if (spotlight) 
		{
			glLightfv(id, GL_SPOT_DIRECTION, spotDirection.GetArray());
			glLightf(id, GL_SPOT_CUTOFF, spotCutOff);
			glLightf(id, GL_SPOT_EXPONENT, spotExponent);
		}
		glLightf(id, GL_CONSTANT_ATTENUATION, attenuation.x);
		glLightf(id, GL_LINEAR_ATTENUATION, attenuation.y);
		glLightf(id, GL_QUADRATIC_ATTENUATION, attenuation.z);
	}

	void SetPosition(float pos[], bool spotFlag)
	{
		position[0] = pos[0];
		position[1] = pos[1];
		position[2] = pos[2];
		spotlight = spotFlag;
		position[3] = (spotlight) ? 1.0f : 0.0f;

		glLightfv(id, GL_POSITION, position);
	}

	void SetPosition(float x, float y, float z, bool spotFlag)
	{
		position[0] = x;
		position[1] = y;
		position[2] = z;
		spotlight = spotFlag;
		position[3] = (spotlight) ? 1.0f : 0.0f;

		glLightfv(id, GL_POSITION, position);
	}

	void SetIntensities(float amb[], float dif[], float spe[])
	{
		ambient[0] = amb[0];
		ambient[1] = amb[1];
		ambient[2] = amb[2];
		ambient[3] = amb[3];

		diffuse[0] = dif[0];
		diffuse[1] = dif[1];
		diffuse[2] = dif[2];
		diffuse[3] = dif[3];

		specular[0] = spe[0];
		specular[1] = spe[1];
		specular[2] = spe[2];
		specular[3] = spe[3];

		glLightfv(id, GL_AMBIENT, ambient);
		glLightfv(id, GL_DIFFUSE, diffuse);
		glLightfv(id, GL_SPECULAR, specular);
	}

	void SetSpotParams(CVector3 direction, float cutoff, float exponent)
	{
		spotDirection = direction;
		spotCutOff = cutoff;
		spotExponent = exponent;

		if (spotlight) 
		{
			glLightfv(id, GL_SPOT_DIRECTION, spotDirection.GetArray());
			glLightf(id, GL_SPOT_CUTOFF, spotCutOff);
			glLightf(id, GL_SPOT_EXPONENT, spotExponent);
		}
	}

	void SetSpotParams(float dirX, float dirY, float dirZ, float cutoff, float exponent)
	{
		spotDirection.Set(dirX, dirY, dirZ);
		spotCutOff = cutoff;
		spotExponent = exponent;

		if (spotlight) 
		{
			glLightfv(id, GL_SPOT_DIRECTION, spotDirection.GetArray());
			glLightf(id, GL_SPOT_CUTOFF, spotCutOff);
			glLightf(id, GL_SPOT_EXPONENT, spotExponent);
		}
	}

	void SetAttenuation(float constant, float linear, float quadratic)
	{
		attenuation.Set(constant, linear, quadratic);

		glLightf(id, GL_CONSTANT_ATTENUATION, attenuation.x);
		glLightf(id, GL_LINEAR_ATTENUATION, attenuation.y);
		glLightf(id, GL_QUADRATIC_ATTENUATION, attenuation.z);
	}

	bool TurnOn()
	{
		GLboolean lighted = true;
		glGetBooleanv(id, &lighted);
	
		if (lighted)
			return false;

		glEnable(id);
		return true;
	}

	void TurnOff()
	{
		glDisable(id);
	}

	void ApplyVectors()
	{
		glLightfv(id, GL_POSITION, position);
		glLightfv(id, GL_SPOT_DIRECTION, spotDirection.GetArray());

		glTranslatef(position[0], position[1], position[2]);
		CVector3 v(spotDirection.y, -spotDirection.x, 0); // (0,0,-1) x SpotDirection = rot. axis
		float angle = (float)acos(-spotDirection.z / spotDirection.Intensity()); // using dot product
		glRotatef(angle * 180.0f / 3.141f, v.x, v.y, v.z);
	}

	private:

	GLenum id;
	bool spotlight;
	float ambient[4];
	float diffuse[4];
	float specular[4];
	float spotCutOff;
	float spotExponent;
	CVector3 spotDirection;
	CVector3 attenuation;
};

#endif