#ifndef _GLCHARACTER_H
#define _GLCHARACTER_H

#include <plib\sl.h>
#include <plib\sm.h>
#include <gl\gl.h>										// Header File For The OpenGL32 Library
#include <gl\glu.h>										// Header File For The GLu32 Library

#include "glbomb.h"
#include "glgrid.h"

#define INITIALSPEED 0.1
#define PI 3.1415926535897932384626433832795
#define OneEightyDivPI (180/PI)

class CFunCharacter
{
public: 
	float radius;	
	float posX;
	float posY;
	float dirX;
	float dirY;
	float dX;
	float dY;
	int numBombItemsTaken;
	int numFireItemsTaken;
	int numMaxFireItemsTaken;
	int numSpeedItemsTaken;
	int numTrapItemsTaken;
	int numWallItemsTaken;
	int numCheeseItemsTaken;
	int numPoints;
	float speed;
	float outOfMazeTime;
	bool hasCheese;


	CFunCharacter(int playerNum, float x, float y, float dx, float dy, float drx, float dry, CFunGrid* grid)
	{
		sprintf(collisionType, "init");
		player = playerNum;
		if (playerNum == 1)
		{
			color[0] = 237;
			color[1] = 74;
			color[2] = 89;
			color[3] = 255;
		}
		else
		{	
			color[0] = 74;
			color[1] = 151;
			color[2] = 237;
			color[3] = 255;
		}
		height = 0.6;
		//radius = 0.2;
		polySubdivisions = 20;
		posX = x;
		posY = y;
		dX = 0;
		dY = 0;
		dirX = dx;
		dirY = dy;
		dirRX = drx;
		dirRY = dry;
		outOfMazeTime = -1;
		hasCheese = false;
		gameGrid = grid;
		numBombItemsTaken = 0;
		numBombs = 1;
		bombsInHand = 1;
		numFireItemsTaken = 0;
		numMaxFireItemsTaken = 0;
		numSpeedItemsTaken = 0;
		numWallItemsTaken = 0;
		numTrapItemsTaken = 0;
		numCheeseItemsTaken = 0;
		numPoints = 0;
		speed = INITIALSPEED;
		maxLegYOffset = 0.01;
		minLegYOffset = -0.13;
		legYOffsetIncrement = 0.01;
		legYOffset = -0.06;
		radius = 0.26;
		upperRadius = 0.23;
		legRadius = 0.1;
		legXOffset = 0.17;
		legZOffset = 0;
		armRadius = 0.06;
		armXOffset = 0.2;
		armYOffset = -upperRadius + 0.1;
		armZOffset = 2 * radius + legZOffset - 0.22;
		backNoseRadius = 0.02;
		noseHeight = 0.05;
		noseZOffset = 2 * radius + legZOffset - 0.1;
		noseYOffset = -upperRadius; 
		frontEarRadius = 0.09;
		backEarRadius = 0.04;
		earHeight = 0.05;
		earZOffset = upperRadius + 2 * radius + legZOffset - 0.05;
		earXOffset = 0.18;
		earYOffset = 0.02;
		earTiltAngle = -100;
		eyeRadius = 0.022;
		eyeZOffset = 2 * radius + legZOffset - 0.04;
		eyeXOffset = 0.07;
		eyeYOffset = -upperRadius + eyeRadius - 0.01;
		whiskerRadius = 0.007;
		whiskerHeight = 0.1;
		whiskerTiltAngle1 = 80;
		whiskerTiltAngle2 = 110;
		whiskerZOffset1 = 2 * radius + legZOffset - 0.09;
		whiskerZOffset2 = 2 * radius + legZOffset - 0.11;
		whiskerXOffset = 0.08;
		whiskerYOffset = -upperRadius - 0.001;
		tailRadius = 0.02;
		tailHeight = 0.1;
		tailYOffset = radius - 0.05;
		tailZOffset = legZOffset + 0.12;
		rotationDegree = 90;
		grid->SetRadius(radius);
		InitBombs();
	}

	void InitSounds(slScheduler* theScheduler)
	{
		soundScheduler = theScheduler;
		soundExplosion = new slSample("sounds/smexp.wav", soundScheduler);
		soundHitWall = new slSample("sounds/wallbounce.wav", soundScheduler);
		soundHitWall->adjustVolume(2.0f);
	}

	void CreateDL()
	{
		playerBodyList = glGenLists(1);
		glNewList(playerBodyList, GL_COMPILE);

		glPushMatrix();
		glTranslatef(0, 0, radius + legZOffset);
		glutSolidSphere(radius, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(0, 0, radius + legZOffset);
		gluCylinder(gluNewQuadric(), radius, upperRadius, radius, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(-armXOffset, armYOffset, armZOffset);
		glutSolidSphere(armRadius, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(armXOffset, armYOffset, armZOffset);
		glutSolidSphere(armRadius, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(0, tailYOffset, tailZOffset);
		glRotatef(-110, 1, 0, 0);
		gluCylinder(gluNewQuadric(), tailRadius, 0, tailHeight, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glEndList();
	
		playerHeadList = glGenLists(2);
		glNewList(playerHeadList, GL_COMPILE);
	
		glPushMatrix();
		glTranslatef(0, 0, 2 * radius + legZOffset);
		glutSolidSphere(upperRadius, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(-earXOffset, earYOffset, earZOffset);
		glRotatef(earTiltAngle, 1, 0, 0);
		glPushMatrix();
		glTranslatef(0, 0, earHeight);
		gluDisk(gluNewQuadric(), 0, backEarRadius, polySubdivisions, polySubdivisions);
		glPopMatrix();
		gluCylinder(gluNewQuadric(), frontEarRadius, backEarRadius, earHeight, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(earXOffset, earYOffset, earZOffset);
		glRotatef(earTiltAngle, 1, 0, 0);
		glPushMatrix();
		glTranslatef(0, 0, earHeight);
		gluDisk(gluNewQuadric(), 0, backEarRadius, polySubdivisions, polySubdivisions);
		glPopMatrix();
		gluCylinder(gluNewQuadric(), frontEarRadius, backEarRadius, earHeight, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glEndList();
		
		playerFaceList = glGenLists(3);
		glNewList(playerFaceList, GL_COMPILE);

		glPushMatrix();
		glTranslatef(-eyeXOffset, eyeYOffset, eyeZOffset);
		glutSolidSphere(eyeRadius, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(eyeXOffset, eyeYOffset, eyeZOffset);
		glutSolidSphere(eyeRadius, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(whiskerXOffset, whiskerYOffset, whiskerZOffset1);
		glRotatef(whiskerTiltAngle1, 0, 1, 0);
		gluCylinder(gluNewQuadric(), whiskerRadius, 0, whiskerHeight, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(whiskerXOffset, whiskerYOffset, whiskerZOffset2);
		glRotatef(whiskerTiltAngle2, 0, 1, 0);
		gluCylinder(gluNewQuadric(), whiskerRadius, 0, whiskerHeight, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(-whiskerXOffset, whiskerYOffset, whiskerZOffset1);
		glRotatef(-whiskerTiltAngle1, 0, 1, 0);
		gluCylinder(gluNewQuadric(), whiskerRadius, 0, whiskerHeight, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(-whiskerXOffset, whiskerYOffset, whiskerZOffset2);
		glRotatef(-whiskerTiltAngle2, 0, 1, 0);
		gluCylinder(gluNewQuadric(), whiskerRadius, 0, whiskerHeight, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(0, noseYOffset, noseZOffset);
		glRotatef(90, 1, 0, 0);
		gluCylinder(gluNewQuadric(), backNoseRadius, 0, noseHeight, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glEndList();
	}

	void InitBombs()
	{
		for (int i = 0; i < 10; i++)
			bombs[i] = new CFunBomb();
	}

	void RenderAllBombs()
	{
		int numBombsInHand = 0;
		for (int i = 0; i < 10; i++)
		{
			if (!bombs[i]->available)
			{
				bool exploding = bombs[i]->Render();
				if (exploding && !bombs[i]->computed)
				{
					soundScheduler->playSample(soundExplosion);
					gameGrid->ComputeExplosion(bombs[i]);
				}
				if (bombs[i]->available)
					gameGrid->ClearBomb(bombs[i]->posX, bombs[i]->posY, i, player);
			}
			else if (i < numBombs)
				numBombsInHand++;
		}
		bombsInHand = numBombsInHand;
	}

	void Render(bool shadow)
	{
		if (dirX > 0)
			rotationDegree = 90 + (atan(dirY/dirX) * OneEightyDivPI);
		else if (dirX < 0)
			rotationDegree = 270 + (atan(dirY/dirX) * OneEightyDivPI);
		else if (dirX == 0 && dirY > 0)
			rotationDegree = 180;
		else if (dirX == 0 && dirY < 0)
			rotationDegree = 0;

		legYOffset += legYOffsetIncrement;
		if (legYOffset > maxLegYOffset)
		{
			legYOffsetIncrement *= -1;
			legYOffset = maxLegYOffset + legYOffsetIncrement;
		}
		else if (legYOffset < minLegYOffset)
		{
			legYOffsetIncrement *= -1;
			legYOffset = minLegYOffset + legYOffsetIncrement;
		}

		glColor4f(0.3, 0.3, 0.3, 1);

		glPushMatrix();
		glTranslatef(posX, posY, 0);
		glRotatef(rotationDegree, 0, 0, 1);

		if (!shadow)
			glColor4f(1, 1, 1, 1);

		glCallList(playerBodyList);

		glPushMatrix();
		glTranslatef(-legXOffset, legYOffset, legRadius);
		glutSolidSphere(legRadius, polySubdivisions, polySubdivisions);
		glPopMatrix();

		glPushMatrix();
		glTranslatef(legXOffset, -0.06 + (-0.06 - legYOffset), legRadius);
		glutSolidSphere(legRadius, polySubdivisions, polySubdivisions);
		glPopMatrix();

		if (!shadow)
			glColor4ub(color[0], color[1], color[2], color[3]);
		
		glCallList(playerHeadList);
		if (!shadow)
			glColor4f(0, 0, 0, 1);

		
		glCallList(playerFaceList);

		if (hasCheese)
		{
			glPushMatrix();
			glTranslatef(0, -radius + 0.05 , radius - 0.07);
			glCallList(gameGrid->items[90]->itemList);
			glPopMatrix();
		}

		glPopMatrix();

		if (!shadow)
			RenderAllBombs();
	}

	void IncrementBomb()
	{
		numBombItemsTaken++;
		if (numBombs < 10)
		{
			numBombs++;
			bombsInHand++;
		}
	}

	void IncrementFireSize()
	{
		numFireItemsTaken++;
		for (int i = 0; i < 10; i++)
		{
			bombs[i]->IncrementSize();
		}
	}

	void MaximizeFireSize()
	{
		numMaxFireItemsTaken++;
		for (int i = 0; i < 10; i++)
		{
			bombs[i]->MaximizeSize();
		}
	}

	void MinimizeFireSize()
	{
		for (int i = 0; i < 10; i++) 
		{
			bombs[i]->MinimizeSize();
		}
	}

	void IncrementSpeed()
	{
		numSpeedItemsTaken++;
		if (speed < 0.45)
			speed += 0.025;
	}

	bool HasTraps()
	{
		if (numTrapItemsTaken <= 0)
			return false;
		else
		{
			numTrapItemsTaken--;
			return true;
		}
	}

	bool HasWalls()
	{
		if (numWallItemsTaken <= 0)
			return false;
		else
		{
			numWallItemsTaken--;
			return true;
		}
	}

	void PlayerKilled()
	{
		if (hasCheese)
			gameGrid->PutDownCheese(posX, posY, player);

		numPoints -= 5;
		if (numPoints < 0)
			numPoints = 0;
		
		if (player == 1)
		{	
			posX = 0.5;
			posY = 0.5;
			dirX = 0;
			dirY = 1;
		}
		else if (player == 2)
		{	
			posX = 14.5;
			posY = 0.5;
			dirX = 0;
			dirY = 1;
		}
		for (int i = 0; i < numBombItemsTaken; i++)
			gameGrid->RecycleItem(1);

		for (i = 0; i < numWallItemsTaken; i++)
			gameGrid->RecycleItem(6);
		
		for (i = 0; i < numTrapItemsTaken; i++)
			gameGrid->RecycleItem(5);
		
		for (i = 0; i < numFireItemsTaken; i++)
			gameGrid->RecycleItem(4);

		for (i = 0; i < numMaxFireItemsTaken; i++)
			gameGrid->RecycleItem(3);

		for (i = 0; i < numSpeedItemsTaken; i++)
			gameGrid->RecycleItem(2);

		numBombItemsTaken = 0;
		numBombs = 1;
		bombsInHand = 1;
		numFireItemsTaken = 0;
		numMaxFireItemsTaken = 0;
		numSpeedItemsTaken = 0;
		numWallItemsTaken = 0;
		numTrapItemsTaken = 0;
		hasCheese = false;
		MinimizeFireSize();
		speed = INITIALSPEED;
	}

	void SetBomb()
	{
		if (bombsInHand > 0)
		{
			int i = 0;
			int found = -1;
			while (i < numBombs && found == -1)
			{
				if (bombs[i]->available)
					found = i;
				i++;
			}

			if (gameGrid->SetBomb(posX, posY, found, player, bombs[found]))
			{
				bombs[found]->SetBomb(posX, posY);	
				bombsInHand--;
			}
		}
	}

	float Absolute(float value)
	{
		if (value >= 0)
			return value;
		else 
			return -value;
	}


	void SetRandomPosition()
	{
		bool set = false;

		while (!set)
		{
			posX = gameGrid->rg->IRandom(0, 14) + 0.5;
			posY = gameGrid->rg->IRandom(0, 14) + 0.5;

			if (gameGrid->IsEmpty(posX, posY, 0) && gameGrid->HasNoObject(posX, posY))
				set = true;	
		}
	}

	bool ForwardBackward()
	{
		dX = 0;
		dY = 0;

		float nextX = posX + dirX * speed;
		float nextY = posY + dirY * speed;
		
		if (hasCheese && ((player == 1 && posX < 1 - radius && posY < 1 - radius) || (player == 2 && posX > 14 + radius && posY < 1 - radius)))
		{
			numCheeseItemsTaken++;
			numPoints += 10;
			hasCheese = false;
			for (int i = 90; i < 95; i++)
			{
				if (gameGrid->items[i]->taken == player)
					gameGrid->items[i]->taken = 3;
			}
		}
		if ((int)posX == 7 && (posY > 8.5 && posY < 9))
		{
			posX = 104.5;
			posY = 100.5;
			dX = 0;
			dY = -(radius + 0.5);
			dirX = 0;
			dirY = 1;
			return false;
		}
		else if ((int)posX == 7 && (posY < 10.5 && posY > 10))
		{
			posX = 104.5;
			posY = 108.5;
			dX = 0;
			dY = radius + 0.5;
			dirX = 0;
			dirY = -1;
			return false;
		}
		else if ((int)posY == 9 && (posX > 6.5 && posX < 7))
		{
			posX = 100.5;
			posY = 104.5;
			dX = -(radius + 0.5);
			dY = 0;
			dirX = 1;
			dirY = 0;
			return false;
		}
		else if ((int)posY == 9 && (posX < 8.5 && posX > 8))
		{
			posX = 108.5;
			posY = 104.5;
			dX = radius + 0.5;
			dY = 0;
			dirX = -1;
			dirY = 0;
			return false;
		}

		if ((nextX >= 0 && nextX < 15 && nextY >= 0 && nextY < 15) &&
			(speed >= 0 && gameGrid->IsEmpty(nextX, nextY, radius) && gameGrid->CheckBomb(posX, posY, nextX, nextY) && gameGrid->CheckWall(posX, posY, nextX, nextY) && (
			  ((dirX == 0 && dirY > 0) && (nextY + radius <= 15))
			|| ((dirX == 0 && dirY < 0) && (nextY - radius >= 0))
			|| ((dirY == 0 && dirX > 0) && (nextX + radius <= 15))
			|| ((dirY == 0 && dirX < 0) && (nextX - radius >= 0))
			
			|| ((dirX > 0 && dirY > 0) && (nextX + radius <= 15 && nextY + radius <= 15))
			|| ((dirX > 0 && dirY < 0) && (nextX + radius <= 15 && nextY - radius >= 0))
			|| ((dirX < 0 && dirY > 0) && (nextX - radius >= 0 && nextY + radius <= 15))
			|| ((dirX < 0 && dirY < 0) && (nextX - radius >= 0 && nextY - radius >= 0))
			))
			||
			(speed < 0 && gameGrid->IsEmpty(nextX, nextY, radius) && gameGrid->CheckBomb(posX, posY, nextX, nextY) && gameGrid->CheckWall(posX, posY, nextX, nextY) && (
			  ((dirX == 0 && dirY < 0) && (nextY + radius <= 15))
			|| ((dirX == 0 && dirY > 0) && (nextY - radius >= 0))
			|| ((dirY == 0 && dirX < 0) && (nextX + radius <= 15))
			|| ((dirY == 0 && dirX > 0) && (nextX - radius >= 0))
			
			|| ((dirX < 0 && dirY < 0) && (nextX + radius <= 15 && nextY + radius <= 15))
			|| ((dirX < 0 && dirY > 0) && (nextX + radius <= 15 && nextY - radius >= 0))
			|| ((dirX > 0 && dirY < 0) && (nextX - radius >= 0 && nextY + radius <= 15))
			|| ((dirX > 0 && dirY > 0) && (nextX - radius >= 0 && nextY - radius >= 0))
			))
			)
		{
			int itemType = gameGrid->CheckItem(nextX, nextY);
			bool getCheese = false;
			switch (itemType)
			{
			case 1:
				IncrementBomb();
				break;
			case 2:
				IncrementSpeed();
				break;
			case 3:
				MaximizeFireSize();
				break;
			case 4:
				IncrementFireSize();
				break;
			case 5:
				numTrapItemsTaken++;
				break;
			case 6:
				numWallItemsTaken++;
				break;
			case 7:
				numPoints++;
				break;
			case 8:
				if (!hasCheese)
				{
					getCheese = true;
					hasCheese = true;
				}
				break;
			default:
				break;
			}
			if (itemType != -1)
			{
				if (itemType != 8 || getCheese)
					gameGrid->ClearItem(nextX, nextY, player);
			}

			if (gameGrid->HasTrap(nextX, nextY) && !gameGrid->HasTrap(posX, posY))
			{
				gameGrid->ClearTrap(nextX, nextY);
				SetRandomPosition();
				dX = 0;
				dY = -2;
				return false;
			}
			else
			{
				dX = nextX - posX;
				dY = nextY - posY;
				posX = nextX;
				posY = nextY;
				return true;
			}
		}
		else if ((nextX >= 0 && nextX < 15 && nextY >= 0 && nextY < 15) && !gameGrid->CheckBomb(posX, posY, nextX, nextY) && !gameGrid->CheckWall(posX, posY, nextX, nextY))
			return true;
		else
		{
			// check positive Y direction
			if (((speed > 0 && dirY > 0) || (speed < 0 && dirY < 0)) &&
				(dirX == 0 || !gameGrid->IsEmpty(posX, nextY, radius)))
			{
				dY = ((int)posY + 1 - radius - 0.01) - posY;
				posY = (int)posY + 1 - radius - 0.01;
			}
			// check negative Y direction
			else if (((speed > 0 && dirY < 0) || (speed < 0 && dirY > 0)) &&
				(dirX == 0 || !gameGrid->IsEmpty(posX, nextY, radius)))
			{
				dY = ((int)posY + radius + 0.01) - posY;
				posY = (int)posY + radius + 0.01;
			}

			// check positive X direction
			if (((speed > 0 && dirX > 0) || (speed < 0 && dirX < 0)) &&
				(dirY == 0 || !gameGrid->IsEmpty(nextX, posY, radius)))
			{
				dX = ((int)posX + 1 - radius - 0.01) - posX;
				posX = (int)posX + 1 - radius - 0.01;
			}
			// check positive X direction
			else if (((speed > 0 && dirX < 0) || (speed < 0 && dirX > 0)) &&
				(dirY == 0 || !gameGrid->IsEmpty(nextX, posY, radius)))
			{
				dX = ((int)posX + radius + 0.01) - posX;
				posX = (int)posX + radius + 0.01;
			}
			return true;
		}
	}

char collisionType[50];

	bool ForwardBackwardMaze(int v)
	{
		float mazeSpeed = 0.1;

		dX = 0;
		dY = 0;

		float nextX = posX + dirX * v * mazeSpeed;
		float nextY = posY + dirY * v * mazeSpeed;

		outOfMazeTime = glutGet(GLUT_ELAPSED_TIME);

		if ((int)posX == 104 && posY < 100.3)
		{
			posX = 7.5;
			posY = 8.5;
			dX = 0;
			dY = 2.0;
			dirX = 0;
			dirY = -1;
			return false;
		}
		else if ((int)posX == 104 && posY > 108.7)
		{
			posX = 7.5;
			posY = 10.5;
			dX = 0;
			dY = -2.0;
			dirX = 0;
			dirY = 1;
			return false;
		}
		else if ((int)posY == 104 && posX < 100.3)
		{
			posX = 6.5;
			posY = 9.5;
			dX = 2.0;
			dY = 0;
			dirX = -1;
			dirY = 0;
			return false;
		}
		else if ((int)posY == 104 && posX > 108.7)
		{
			posX = 8.5;
			posY = 9.5;
			dX = -2.0;
			dY = 0;
			dirX = 1;
			dirY = 0;
			return false;
		}


		float incrX = dirX * v * 0.01;
		float incrY = dirY * v * 0.01;

		bool hitWall = false;

		bool hWall1 = false;
		bool hWall2 = false;
		bool vWall1 = false;
		bool vWall2 = false;

		CFunMaze* theMaze = gameGrid->maze;
		int mazeSize = theMaze->mazeSize;

		float x[72];
		float y[72];

		float initDegree;

		int gridIndex = ((int)(posX-100)) + (((int)(posY-100)) * mazeSize);
		int collisionDetect = -1;
		//up
		if ((v > 0 && dirX == 0 && dirY > 0) || (v < 0 && dirX == 0 && dirY < 0))
		{
			//left
			if (posX - (int)posX < radius + 0.025)
			{
				hWall1 = theMaze->wallHorizontal[gridIndex];
				if ((gridIndex - 1) % mazeSize != 8)
					hWall2 = theMaze->wallHorizontal[gridIndex - 1];
				else
					hWall2 = true;
				if (gridIndex + mazeSize - 1 < 81 && (gridIndex + mazeSize - 1) % mazeSize != 8)
					vWall2 = theMaze->wallVertical[gridIndex + mazeSize - 1];
				else
					vWall2 = true;
				vWall1 = false;
				collisionDetect = 1;
				sprintf(collisionType, "up-left: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}
			//right
			else if ((int)posX + 1 - posX < radius + 0.025)
			{
				hWall1 = theMaze->wallHorizontal[gridIndex];
				if ((gridIndex + 1) % mazeSize != 0)
					hWall2 = theMaze->wallHorizontal[gridIndex + 1];
				else
					hWall2 = true;
				vWall1 = false;
				if (gridIndex + mazeSize < 81)
					vWall2 = theMaze->wallVertical[gridIndex + mazeSize];
				else
					vWall2 = true;
				collisionDetect = 2;
				sprintf(collisionType, "up-right: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}
			//middle
			else
			{
				hWall1 = theMaze->wallHorizontal[gridIndex];
				hWall2 = false;
				vWall1 = false;
				vWall2 = false;
				collisionDetect = 3;
				sprintf(collisionType, "up-middle: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}

			initDegree = 0;
		}
		//down
		else if ((v > 0 && dirX == 0 && dirY < 0) || (v < 0 && dirX == 0 && dirY > 0))
		{
			//left
			if (posX - (int)posX < radius + 0.025)
			{
				hWall1 = theMaze->wallHorizontal[gridIndex - mazeSize];
				if (gridIndex - mazeSize >= 0 && (gridIndex - mazeSize - 1) % mazeSize != 8)
				{
					hWall2 = theMaze->wallHorizontal[gridIndex - mazeSize - 1];
					vWall2 = theMaze->wallVertical[gridIndex - mazeSize - 1];
				}
				else
				{
					hWall2 = true;
					vWall2 = true;
				}
				vWall1 = false;
				collisionDetect = 4;
				sprintf(collisionType, "down-left: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}
			//right
			else if ((int)posX + 1 - posX < radius + 0.025)
			{
				if (gridIndex - mazeSize >= 0)
				{
					hWall1 = theMaze->wallHorizontal[gridIndex - mazeSize];
					vWall2 = theMaze->wallVertical[gridIndex - mazeSize];
				}
				else
				{
					hWall1 = true;
					vWall2 = true;
				}
				if (gridIndex - mazeSize + 1 >= 0 && (gridIndex - mazeSize + 1) % mazeSize != 0)
					hWall2 = theMaze->wallHorizontal[gridIndex - mazeSize + 1];
				else
					hWall2 = true;
				vWall1 = false;
				collisionDetect = 5;
				sprintf(collisionType, "down-right: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
				
			}
			//middle
			else
			{
				if (gridIndex - mazeSize >= 0)
					hWall1 = theMaze->wallHorizontal[gridIndex - mazeSize];
				else
					hWall1 = true;
				hWall2 = false;
				vWall1 = false;
				vWall2 = false;
				collisionDetect = 6;
				sprintf(collisionType, "down-middle: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}

			initDegree = PI;
		}
		//right
		else if ((v > 0 && dirY == 0 && dirX > 0) || (v < 0 && dirY == 0 && dirX < 0))
		{
			//down
			if (posY - (int)posY < radius + 0.025)
			{
				hWall1 = false;
				if (gridIndex - mazeSize + 1 >= 0 && (gridIndex - mazeSize + 1) % mazeSize != 0)
					hWall2 = theMaze->wallHorizontal[gridIndex - mazeSize + 1];
				else
					hWall2 = true;
				vWall1 = theMaze->wallVertical[gridIndex];
				if (gridIndex - mazeSize >= 0)
					vWall2 = theMaze->wallVertical[gridIndex - mazeSize];
				else
					vWall2 = true;
				collisionDetect = 7;
				sprintf(collisionType, "right-down: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}
			//up
			else if ((int)posY + 1 - posY < radius + 0.025)
			{
				hWall1 = false;
				if ((gridIndex + 1) % mazeSize != 0)
					hWall2 = theMaze->wallHorizontal[gridIndex + 1];
				else
					hWall2 = true;
				vWall1 = theMaze->wallVertical[gridIndex];
				if (gridIndex + mazeSize < 81)
					vWall2 = theMaze->wallVertical[gridIndex + mazeSize];
				else 
					vWall2 = true;
				collisionDetect = 8;
				sprintf(collisionType, "right-up: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}
			//middle
			else
			{
				hWall1 = false;
				hWall2 = false;
				vWall1 = theMaze->wallVertical[gridIndex];
				vWall2 = false;
				collisionDetect = 9;
				sprintf(collisionType, "right-middle: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}

			initDegree = -PI/2;
		}
		//left
		else if ((v > 0 && dirY == 0 && dirX < 0) || (v < 0 && dirY == 0 && dirX > 0))
		{
			//down
			if (posY - (int)posY < radius + 0.025)
			{
				hWall1 = false;
				if (gridIndex - mazeSize >= 0  && (gridIndex - mazeSize - 1) % mazeSize != 8)
				{
					hWall2 = theMaze->wallHorizontal[gridIndex - mazeSize - 1];
					vWall2 = theMaze->wallVertical[gridIndex - mazeSize - 1];
				}
				else
				{
					hWall2 = true;
					vWall2 = true;
				}
				if ((gridIndex - 1) % mazeSize != 8)
					vWall1 = theMaze->wallVertical[gridIndex - 1];
				else
					vWall1 = true;
				collisionDetect = 10;
				sprintf(collisionType, "left-down: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}
			//up
			else if ((int)posY + 1 - posY < radius + 0.025)
			{
				hWall1 = false;
				if ((gridIndex - 1) % mazeSize != 8)
				{
					hWall2 = theMaze->wallHorizontal[gridIndex - 1];
					vWall1 = theMaze->wallVertical[gridIndex - 1];
				}
				else
				{
					hWall2 = true;
					vWall1 = true;
				}
				if (gridIndex + mazeSize - 1 < 81 && (gridIndex + mazeSize - 1) % mazeSize != 8)
					vWall2 = theMaze->wallVertical[gridIndex + mazeSize - 1];
				else
					vWall2 = true;
				collisionDetect = 11;
				sprintf(collisionType, "left-up: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}
			//middle
			else
			{
				hWall1 = false;
				hWall2 = false;
				if ((gridIndex - 1) % mazeSize != 8)
					vWall1 = theMaze->wallVertical[gridIndex - 1];
				else
					vWall1 = true;
				vWall2 = false;
				collisionDetect = 12;
				sprintf(collisionType, "left-middle: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}

			initDegree = PI/2;
		}
		//right-up
		else if ((v > 0 && dirX > 0 && dirY > 0) || (v < 0 && dirX < 0 && dirY < 0))
		{
			//left
			if (posX < (int)posX + 0.5)
			{
				hWall1 = theMaze->wallHorizontal[gridIndex];
				vWall1 = false;
				if ((gridIndex - 1) % mazeSize != 8)
					hWall2 = theMaze->wallHorizontal[gridIndex - 1];
				else
					hWall2 = true;
				if (gridIndex + mazeSize - 1 < 81 && (gridIndex + mazeSize - 1) % mazeSize != 8)
					vWall2 = theMaze->wallVertical[gridIndex + mazeSize - 1];
				else
					vWall2 = true;
				collisionDetect = 13;
				sprintf(collisionType, "right-up-left: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}
			//right
			else
			{
				hWall1 = theMaze->wallHorizontal[gridIndex];
				vWall1 = theMaze->wallVertical[gridIndex];
				//up
				if (posY >= (int)posY + 0.5)
				{
					if ((gridIndex + 1) % mazeSize != 0)
						hWall2 = theMaze->wallHorizontal[gridIndex + 1];
					else
						hWall2 = true;
					if (gridIndex + mazeSize < 81)
						vWall2 = theMaze->wallVertical[gridIndex + mazeSize];
					else
						vWall2 = true;
				collisionDetect = 14;
				sprintf(collisionType, "right-up-right-up: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
				}
				//down
				else
				{
					if (gridIndex - mazeSize >= 0 && (gridIndex - mazeSize + 1) % mazeSize != 0) 
						hWall2 = theMaze->wallHorizontal[gridIndex - mazeSize + 1];
					else
						hWall2 = true;
					if (gridIndex - mazeSize >= 0)
						vWall2 = theMaze->wallVertical[gridIndex - mazeSize];
					else
						vWall2 = true;

					hWall1 = false;
					collisionDetect = 15;
				sprintf(collisionType, "right-up-right-down: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
				}
			}

			initDegree = atan(dirY/dirX) - PI/2;
		}
		//right-down
		else if ((v > 0 && dirX > 0 && dirY < 0) || (v < 0 && dirX < 0 && dirY > 0))
		{
			//left
			if (posX < (int)posX + 0.5)
			{
				if (gridIndex - mazeSize >= 0)
					hWall1 = theMaze->wallHorizontal[gridIndex - mazeSize];
				else
					hWall1 = true;
				if (gridIndex - mazeSize >= 0 && (gridIndex - mazeSize - 1) % mazeSize != 8)
				{
					hWall2 = theMaze->wallHorizontal[gridIndex - mazeSize - 1];
					vWall2 = theMaze->wallVertical[gridIndex - mazeSize - 1];
				}
				else
				{
					hWall2 = true;
					vWall2 = true;
				}
				vWall1 = false;
				collisionDetect = 16;
				sprintf(collisionType, "right-down-left: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}
			//right
			else
			{				
				if (gridIndex - mazeSize >= 0)
					hWall1 = theMaze->wallHorizontal[gridIndex - mazeSize];
				else
					hWall1 = true;
				vWall1 = theMaze->wallVertical[gridIndex];
				//down
				if (posY < (int)posY + 0.5)
				{
					if (gridIndex - mazeSize >= 0 && (gridIndex - mazeSize + 1) % mazeSize != 0)
						hWall2 = theMaze->wallHorizontal[gridIndex - mazeSize + 1];
					else
						hWall2 = true;
					if (gridIndex - mazeSize >= 0)
						vWall2 = theMaze->wallVertical[gridIndex - mazeSize];
					else
						vWall2 = true;
				collisionDetect = 17;
				sprintf(collisionType, "right-down-right-down: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
				}
				//up
				else
				{
					if ((gridIndex + 1) % mazeSize != 0)
						hWall2 = theMaze->wallHorizontal[gridIndex + 1];
					else 
						hWall2 = true;
					if (gridIndex + mazeSize < 81)
						vWall2 = theMaze->wallVertical[gridIndex + mazeSize];
					else
						vWall2 = true;
				
					hWall1 = false;
					collisionDetect = 18;
				sprintf(collisionType, "right-down-right-up: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
				}
			}
			initDegree = atan(dirY/dirX) - PI/2;
		}
		//left-up
		else if ((v > 0 && dirX < 0 && dirY > 0) || (v < 0 && dirX > 0 && dirY < 0))
		{
			//right
			if (posX >= (int)posX + 0.5)
			{
				hWall1 = theMaze->wallHorizontal[gridIndex];
				if ((gridIndex + 1) % mazeSize != 0)	
					hWall2 = theMaze->wallHorizontal[gridIndex + 1];
				else
					hWall2 = true;
				vWall1 = false;
				if (gridIndex + mazeSize < 81)
					vWall2 = theMaze->wallVertical[gridIndex + mazeSize];
				else
					vWall2 = true;
				collisionDetect = 19;
				sprintf(collisionType, "left-up-right: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}
			//left
			else
			{
				hWall1 = theMaze->wallHorizontal[gridIndex];
				if ((gridIndex - 1) % mazeSize != 8)
					vWall1 = theMaze->wallVertical[gridIndex - 1];
				else
					vWall1 = true;
				//up
				if (posY >= (int)posY + 0.5)
				{
					if ((gridIndex - 1) % mazeSize != 8)
						hWall2 = theMaze->wallHorizontal[gridIndex - 1];
					else
						hWall2 = true;
					if (gridIndex + mazeSize - 1 < 81 && (gridIndex + mazeSize - 1) % mazeSize != 8)
						vWall2 = theMaze->wallVertical[gridIndex + mazeSize - 1];
					else
						vWall2 = true;
				collisionDetect = 20;
				sprintf(collisionType, "left-up-left-up: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
				}
				//down
				else
				{
					if (gridIndex - mazeSize >= 0 && (gridIndex - mazeSize - 1) % mazeSize != 8)
					{
						hWall2 = theMaze->wallHorizontal[gridIndex - mazeSize - 1];
						vWall2 = theMaze->wallHorizontal[gridIndex - mazeSize - 1];
					}
					else
					{
						hWall2 = true;
						vWall2 = true;
					}
					
					hWall1 = false;
					collisionDetect = 21;
				sprintf(collisionType, "left-up-left-down: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
				}
			}
			
			initDegree = PI + atan(dirY/dirX) - PI/2;
		}
		//left-down
		else if ((v > 0 && dirX < 0 && dirY < 0) || (v < 0 && dirX > 0 && dirY > 0))
		{
			//left
			if (posX < (int)posX + 0.5)
			{
				if (gridIndex - mazeSize >= 0)
					hWall1 = theMaze->wallHorizontal[gridIndex - mazeSize];
				else
					hWall1 = true;
				if ((gridIndex - 1) % mazeSize != 8)
					vWall1 = theMaze->wallVertical[gridIndex - 1];
				else
					vWall1 = true;
			
				//down
				if (posY < (int)posY + 0.5)
				{
					if (gridIndex - mazeSize >= 0 && (gridIndex - mazeSize) % mazeSize != 8)
					{
						vWall2 = theMaze->wallVertical[gridIndex - mazeSize - 1];
						hWall2 = theMaze->wallHorizontal[gridIndex - mazeSize - 1];
					}
					else
					{
						vWall2 = true;
						hWall2 = true;
					}
				collisionDetect = 22;
				sprintf(collisionType, "left-down-left-down: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
				}
				//up
				else
				{
					if ((gridIndex - 1) % mazeSize != 8)
						hWall2 = theMaze->wallHorizontal[gridIndex - 1];
					else
						hWall2 = true;
					if (gridIndex + mazeSize - 1 < 81 && (gridIndex + mazeSize - 1) % mazeSize != 8)
						vWall2 = theMaze->wallVertical[gridIndex + mazeSize - 1];
					else
						vWall2 = true;
		
					hWall1 = false;		
					collisionDetect = 23;
				sprintf(collisionType, "left-down-left-up: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
				}
			}
			//right
			else
			{
				if (gridIndex - mazeSize >= 0)
					hWall1 = theMaze->wallHorizontal[gridIndex - mazeSize];
				else
					hWall1 = true;
				vWall1 = false;
				if (gridIndex - mazeSize >= 0)
					vWall2 = theMaze->wallVertical[gridIndex - mazeSize];
				else
					vWall2 = true;
				if (gridIndex - mazeSize >= 0 && (gridIndex - mazeSize + 1) % mazeSize != 0)
					hWall2 = theMaze->wallHorizontal[gridIndex - mazeSize + 1];
				else
					hWall2 = true;
				collisionDetect = 24;
				sprintf(collisionType, "left-down-right: h1:%d, v1:%d, h2:%d, v2:%d", hWall1, vWall1, hWall2, vWall2);
			}

			initDegree = PI + atan(dirY/dirX) - PI/2;
		}


		initDegree = 0;
		for (int i = 0; i < 72; i++)
		{
			x[i] = posX + radius * cos(initDegree);
			y[i] = posY + radius * sin(initDegree);
			initDegree += PI/36;
		}

		float newX = posX;
		float newY = posY;

		float tempX;
		float tempY;

		while (!hitWall && ((dirX == 0 && Absolute(newY - nextY) >= 0.01) || (dirY == 0 && Absolute(newX - nextX) >= 0.01) || (Absolute(newX - nextX) >= 0.01 || Absolute(newY - nextY) >= 0.01)))
		{
			for (i = 0; i < 72; i++)
			{
				tempX = x[i] + incrX;
				tempY = y[i] + incrY;

				switch(collisionDetect)
				{
				case 1: //up-left
					hitWall = ( 
						(hWall1 && tempY + 0.025 > (int)(posY + 1))  
						|| 
						(hWall2 && tempY + 0.025 > (int)(posY + 1)) 
						|| 
						(vWall2 && tempY + 0.025 > (int)(posY + 1) && tempX - 0.025 < (int)posX)
						);
					break;
				case 2: //up-right
					hitWall = ( 
						(hWall1 && tempY + 0.025 > (int)(posY + 1))  
						|| 
						(hWall2 && tempY + 0.025 > (int)(posY + 1)) 
						|| 
						(vWall2 && tempY + 0.025 > (int)(posY + 1) && tempX + 0.025 > (int)(posX + 1))
						);
					break;
				case 3: //up-midle
					hitWall = ( 
						(hWall1 && tempY + 0.025 > (int)(posY + 1))  
						);
					break;
				case 4: //down-left
					hitWall = ( 
						(hWall1 && tempY - 0.025 < (int)posY)  
						|| 
						(hWall2 && tempY - 0.025 < (int)posY) 
						|| 
						(vWall2 && tempY - 0.025 < (int)posY && tempX - 0.025 < (int)posX)
						);
					break;
				case 5: //down-right
					hitWall = ( 
						(hWall1 && tempY - 0.025 < (int)posY)  
						|| 
						(hWall2 && tempY - 0.025 < (int)posY) 
						|| 
						(vWall2 && tempY - 0.025 < (int)posY && tempX + 0.025 > (int)(posX + 1))
						);
					break;
				case 6: //down-middle
					hitWall = ( 
						(hWall1 && tempY - 0.025 < (int)posY)  
						);
					break;
				case 7: //right-down
					hitWall = ( 
						(vWall1 && tempX + 0.025 > (int)(posX + 1)) 
						|| 
						(hWall2 && tempX + 0.025 > (int)(posX + 1) && tempY - 0.025 < (int)posY) 
						|| 
						(vWall2 && tempX + 0.025 > (int)(posX + 1))
						);
					break;
				case 8: //right-up
					hitWall = ( 
						(vWall1 && tempX + 0.025 > (int)(posX + 1)) 
						|| 
						(hWall2 && tempX + 0.025 > (int)(posX + 1) && tempY + 0.025 > (int)(posY + 1)) 
						|| 
						(vWall2 && tempX + 0.025 > (int)(posX + 1))
						);
					break;
				case 9: //right-middle
					hitWall = ( 
						(vWall1 && tempX + 0.025 > (int)(posX + 1)) 
						);
					break;
				case 10: //left-down
					hitWall = ( 
						(vWall1 && tempX - 0.025 < (int)posX) 
						|| 
						(hWall2 && tempX - 0.025 < (int)posX && tempY - 0.025 < (int)posY) 
						|| 
						(vWall2 && tempX - 0.025 < (int)posX)
						);
					break;
				case 11: //left-up
					hitWall = ( 
						(vWall1 && tempX - 0.025 < (int)posX) 
						|| 
						(hWall2 && tempX - 0.025 < (int)posX && tempY + 0.025 > (int)(posY + 1)) 
						|| 
						(vWall2 && tempX - 0.025 < (int)posX)
						);
					break;
				case 12: //left-middle
					hitWall = ( 
						(vWall1 && tempX - 0.025 < (int)posX) 
						);
					break;
				case 13: //right-up-left
					hitWall = ( 
						(hWall1 && tempY + 0.025 > (int)(posY + 1) && tempX > (int)posX)
						|| 
						(hWall2 && tempY + 0.025 > (int)(posY + 1) && tempX < (int)posX) 
						|| 
						(vWall2 && tempY + 0.025 > (int)(posY + 1) && tempX < (int)posX)
						);
					break;
				case 14: //right-up-right-up
					hitWall = ( 
						(hWall1 && tempY + 0.025 > (int)(posY + 1) && tempX < (int)(posX + 1))
						|| 
						(vWall1 && tempY < (int)(posY + 1) && tempX + 0.025 > (int)(posX + 1)) 
						|| 
						(hWall2 && tempY + 0.025 > (int)(posY + 1) && tempX > (int)(posX + 1)) 
						|| 
						(vWall2 && tempY + 0.025 > (int)(posY + 1) && tempX > (int)(posX + 1))
						);
					break;
				case 15: //right-up-right-down
					hitWall = ( 
						(vWall1 && tempY > (int)posY && tempX + 0.025 > (int)(posX + 1)) 
						|| 
						(hWall2 && tempY - 0.025 < (int)posY && tempX + 0.025 > (int)(posX + 1)) 
						|| 
						(vWall2 && tempY - 0.025 < (int)posY && tempX + 0.025 > (int)(posX + 1))
						);
					break;
				case 16: //right-down-left
					hitWall = ( 
						(hWall1 && tempY - 0.025 < (int)posY && tempX > (int)posX)
						|| 
						(hWall2 && tempY - 0.025 < (int)posY && tempX < (int)posX) 
						|| 
						(vWall2 && tempY - 0.025 < (int)posY && tempX < (int)posX)
						);
					break;
				case 17: //right-down-right-down
					hitWall = ( 
						(hWall1 && tempY - 0.025 < (int)posY && tempX < (int)(posX + 1))
						|| 
						(vWall1 && tempY > (int)posY && tempX + 0.025 > (int)(posX + 1)) 
						|| 
						(hWall2 && tempY - 0.025 < (int)posY && tempX > (int)(posX + 1)) 
						|| 
						(vWall2 && tempY - 0.025 < (int)posY && tempX > (int)(posX + 1))
						);
					break;
				case 18: //right-down-right-up
					hitWall = (
						(vWall1 && tempY < (int)(posY + 1) && tempX + 0.025 > (int)(posX + 1)) 
						|| 
						(hWall2 && tempY > (int)(posY + 1) && tempX > (int)(posX + 1)) 
						|| 
						(vWall2 && tempY > (int)(posY + 1) && tempX > (int)(posX + 1))
						);
					break;
				case 19: //left-up-right
					hitWall = ( 
						(hWall1 && tempY + 0.025 > (int)(posY + 1) && tempX < (int)(posX + 1))
						|| 
						(hWall2 && tempY + 0.025 > (int)(posY + 1) && tempX + 0.025 > (int)(posX + 1)) 
						|| 
						(vWall2 && tempY + 0.025 > (int)(posY + 1) && tempX + 0.025 > (int)(posX + 1))
						);
					break;
				case 20: //left-up-left-up
					hitWall = ( 
						(hWall1 && tempY + 0.025 > (int)(posY + 1) && tempX < (int)(posX + 1))
						|| 
						(vWall1 && tempY < (int)(posY + 1) && tempX - 0.025 < (int)posX) 
						|| 
						(hWall2 && tempY + 0.025 > (int)(posY + 1) && tempX < (int)posX) 
						|| 
						(vWall2 && tempY + 0.025 > (int)(posY + 1) && tempX - 0.025 < (int)posX)
						);
					break;
				case 21: //left-up-left-down
					hitWall = ( 
						(vWall1 && tempY > (int)posY && tempX - 0.025 < (int)posX) 
						|| 
						(hWall2 && tempY - 0.025 < (int)posY && tempX - 0.025 < (int)posX) 
						|| 
						(vWall2 && tempY - 0.025 < (int)posY && tempX - 0.025 < (int)posX)
						);
					break;
				case 22: //left-down-left-down
					hitWall = ( 
						(hWall1 && tempY - 0.025 < (int)posY && tempX > (int)posX)
						|| 
						(vWall1 && tempY > (int)posY && tempX - 0.025 > (int)posX) 
						|| 
						(hWall2 && tempY - 0.025 < (int)posY && tempX - 0.025 < (int)posX) 
						|| 
						(vWall2 && tempY - 0.025 < (int)posY && tempX - 0.025 < (int)posX)
						);
					break;
				case 23: //left-down-left-up
					hitWall = ( 
						(vWall1 && tempY < (int)(posY + 1) && tempX - 0.025 > (int)posX)
						|| 
						(hWall2 && tempY + 0.025 > (int)(posY + 1) && tempX - 0.025 < (int)posX) 
						|| 
						(vWall2 && tempY + 0.025 > (int)(posY + 1) && tempX - 0.025 < (int)posX)
						);
					break;
				case 24: //left-down-right
					hitWall = ( 
						(hWall1 && tempY - 0.025 < (int)posY && tempX + 0.025 < (int)(posX + 1)) 
						|| 
						(hWall2 && tempY - 0.025 <(int)posY && tempX + 0.025 > (int)(posX + 1)) 
						|| 
						(vWall2 && tempY- 0.025 < (int)posY && tempX + 0.025 > (int)(posX + 1))
						);
					break;
				default:
					hitWall = false;
					break;
				}

				if (!hitWall)
				{
					x[i] = tempX;
					y[i] = tempY;
				}
				else
					break;
			}

			if (!hitWall)
			{
				newX += incrX;
				newY += incrY;
			}
		}

		if (hitWall)
			soundScheduler->playSample(soundHitWall);

		dX = newX - posX;
		dY = newY - posY;

		posX = newX;
		posY = newY;

		if (!hasCheese && gameGrid->CheckCheese(posX, posY, player))
			hasCheese = true;

		return true;
	}


	void LeftRight()
	{
		dX = 0;
		dY = 0;

		float nextX = posX + dirRX * speed;
		float nextY = posY + dirRY * speed;

		if (
			(speed >= 0 && gameGrid->IsEmpty(nextX, nextY, radius) && gameGrid->CheckBomb(posX, posY, nextX, nextY) && (
			  ((dirRX == 0 && dirRY > 0) && (nextY + radius <= 15))
			|| ((dirRX == 0 && dirRY < 0) && (nextY - radius >= 0))
			|| ((dirRY == 0 && dirRX > 0) && (nextX + radius <= 15))
			|| ((dirRY == 0 && dirRX < 0) && (nextX - radius >= 0))
			
			|| ((dirRX > 0 && dirRY > 0) && (nextX + radius <= 15 && nextY + radius <= 15))
			|| ((dirRX > 0 && dirRY < 0) && (nextX + radius <= 15 && nextY - radius >= 0))
			|| ((dirRX < 0 && dirRY > 0) && (nextX - radius >= 0 && nextY + radius <= 15))
			|| ((dirRX < 0 && dirRY < 0) && (nextX - radius >= 0 && nextY - radius >= 0))
			))
			||
			(speed < 0 && gameGrid->IsEmpty(nextX, nextY, radius) && gameGrid->CheckBomb(posX, posY, nextX, nextY) &&(
			  ((dirRX == 0 && dirRY < 0) && (nextY + radius <= 15))
			|| ((dirRX == 0 && dirRY > 0) && (nextY - radius >= 0))
			|| ((dirRY == 0 && dirRX < 0) && (nextX + radius <= 15))
			|| ((dirRY == 0 && dirRX > 0) && (nextX - radius >= 0))
			
			|| ((dirRX < 0 && dirRY < 0) && (nextX + radius <= 15 && nextY + radius <= 15))
			|| ((dirRX < 0 && dirRY > 0) && (nextX + radius <= 15 && nextY - radius >= 0))
			|| ((dirRX > 0 && dirRY < 0) && (nextX - radius >= 0 && nextY + radius <= 15))
			|| ((dirRX > 0 && dirRY > 0) && (nextX - radius >= 0 && nextY - radius >= 0))
			))
			)
		{
			dX = nextX - posX;
			dY = nextY - posY;
			posX = nextX;
			posY = nextY;
		}
		else if (!gameGrid->CheckBomb(posX, posY, nextX, nextY))
			return;
		else
		{
			// check positive Y direction
			if (((speed > 0 && dirRY > 0) || (speed < 0 && dirRY < 0)) &&
				(dirRX == 0 || !gameGrid->IsEmpty(posX, nextY, radius)))
			{
				dY = ((int)posY + 1 - radius - 0.01) - posY;
				posY = (int)posY + 1 - radius - 0.01;
			}
			// check negative Y direction
			else if (((speed > 0 && dirRY < 0) || (speed < 0 && dirRY > 0)) &&
				(dirRX == 0 || !gameGrid->IsEmpty(posX, nextY, radius)))
			{
				dY = ((int)posY + radius + 0.01) - posY;
				posY = (int)posY + radius + 0.01;
			}

			// check positive X direction
			if (((speed > 0 && dirRX > 0) || (speed < 0 && dirRX < 0)) &&
				(dirRY == 0 || !gameGrid->IsEmpty(nextX, posY, radius)))
			{
				dX = ((int)posX + 1 - radius - 0.01) - posX;
				posX = (int)posX + 1 - radius - 0.01;
			}
			// check positive X direction
			else if (((speed > 0 && dirRX < 0) || (speed < 0 && dirRX > 0)) &&
				(dirRY == 0 || !gameGrid->IsEmpty(nextX, posY, radius)))
			{
				dX = ((int)posX + radius + 0.01) - posX;
				posX = (int)posX + radius + 0.01;
			}
		}
	}


	void Rotate(CVector3 direction, CVector3 rightDirection)
	{
		direction.z = 0;
		direction.Normalize();
		dirX = direction.x;
		dirY = direction.y;
		rightDirection.z = 0;
		rightDirection.Normalize();
		dirRX = rightDirection.x;
		dirRY = rightDirection.y;
	}

private:
	int player;
	float color[4];
	float height;
	float polySubdivisions;
	float dirRX;
	float dirRY;
	CFunGrid* gameGrid;
	CFunBomb* bombs[10];
	int bombsInHand;
	int numBombs;
	float upperRadius;
	float legRadius;
	float maxLegYOffset;
	float minLegYOffset;
	float legYOffsetIncrement;
	float legYOffset;
	float legXOffset;
	float legZOffset;
	float armRadius;
	float armXOffset;
	float armYOffset;
	float armZOffset;
	float backNoseRadius;
	float noseHeight;
	float noseZOffset;
	float noseYOffset; 
	float frontEarRadius;
	float backEarRadius;
	float earHeight;
	float earZOffset;
	float earXOffset;
	float earYOffset;
	float earTiltAngle;
	float eyeRadius;
	float eyeZOffset;
	float eyeXOffset;
	float eyeYOffset;
	float whiskerRadius;
	float whiskerHeight;
	float whiskerTiltAngle1;
	float whiskerTiltAngle2;
	float whiskerZOffset1;
	float whiskerZOffset2;
	float whiskerXOffset;
	float whiskerYOffset;
	float tailRadius;
	float tailHeight;
	float tailYOffset;
	float tailZOffset;
	float rotationDegree;
	GLuint playerBodyList;
	GLuint playerHeadList;
	GLuint playerFaceList;

	slScheduler* soundScheduler;
	slSample* soundExplosion;
	slSample* soundHitWall;
};

#endif